// ********************************************************************************************************************************************
// Pepper card types recognition                                         treibauf AG, Switzerland, 04.11.2020 // updated with Pepper rev. 23663
// ********************************************************************************************************************************************
// https://en.wikipedia.org/wiki/Payment_card_number
// 
// This file enables Pepper to recognise the processed card types.
// In order for Pepper to use this file, it has to be placed in the working directory of Pepper, and must be named 'cardtypes.pep'.
// Its content is loaded into Pepper only once. If changes are made in this file, Pepper should be restarted to make the changes effective.
//
// Comment lines (starting with "//") and empty lines (CR+LF) will not be interpreted.
//
// Each line represents a card type recognition entry. It consists of terminal type, input, output and cardname.
// The format for each entry is: 'terminaltype.[input]=output{,cardname};'
// Pepper internally converts everything to lower-case and thus is not case sensitive.
// All leading and trailing spaces of an entry are stripped off.
// Prefixes can be used as input. A prefix input has the following syntax: '+preYES,-preNO,...'.
// A prefix 'preYES' is searched, and it matches only if there are no 'preNO' that matches too.
// Example: The key is '6.[+591,-59184,-59185]=302;' ==> '59186324324' will match, whereas '59184324234' won't match.
// The order you enter the prefix keys is relevant; the first match will be applied.
// Multiple card type recognition entries with the same result could be combined; again the first match will be applied.
//
// Each terminal type has a specific card recognition, based on different keys.
// For this reason, the comments concerning the method for the card recognition must be strictly observed.
//
// Values for terminal types:
//   0 (CH VEZ ASCOM),
//   1 (CH VEZ SIX PAYMENT SERVICES),
//   2 (CH VEZ EPSYS),
//   3 (CH VEZ CASH),
//   4 (CH VEZ CCREDIT),
//   5 (AT FIRST DATA),
//   6 (SI BANKART),
//   7 (CH/AT/SI/LU/CZ/HU SIX PAYMENT SERVICES MPD),
//   8 (DE ZVT GENERIC),
//   9 (DE INGENICO U16),
//  10 (DE HYPERCOM ARTEMA),
//  11 (CH ACCARDA VIRTUAL CLIENT),
//  12 (BE ATOS WORLDLINE),
//  13 (SIMPATICO EFT-SIMULATOR),
//  14 (CH VEZ PAYSYS),
//  15 (CH ADUNO OPTIMUM),
//  16 (US VERIFONE PCCHARGE),
//  17 (CH VEZ+ SIX PAYMENT SERVICES),
//  18 (DE HYPERCOM/THALES EFT Library),
//  19 (SI SIX PAYMENT SERVICES MPD (deprecated) ),
//  20 (CH INGENICO CRP 2.42),
//  21 (AT/LU SIX PAYMENT SERVICES MPD (deprecated) ),
//  22 (SI CCV ZVT),
//  23 (CZ INGENICO CZESKA SPORITELNA),
//  24 (FR INGENICO CONCERT),
//  25 (DE INGENICO U32),
//  26 (CH ADUNO VERDI),
//  27 (CH ADUNO BLADE)
//  28 (NL BANKSYS XENTA VIC-NL),
//  29 (AT CARD COMPLETE SMART),
//  30 (DE ELPAY),
//  31 (IT INGENICO),
//  32 (ES INGENICO),
//  33 (PL FIRST DATA / POLCARD),
//  34 (AT PAYLIFE ARTEMA HYBRID),
//  35 (CH CCV JERONIMO CCVJIL.DLL),
//  36 (FR/ES/UK INGENICO C3 / AXIS),
//  37 (CH INNOCARD ARTEMA),
//  38 (CH INNOCARD OPTIMUM),
//  39 (DE PAYSYS),
//  40 (SK SLOVENSKA SPORITELNA),
//  41 (CH MFGROUP PAYMENT SERVER GATEWAY),
//  42 (DE/AT/LU EASYCASH PREPAYPLUS),
//  43 (DE/AT/CH CARDS & SYSTEMS IBRO30),
//  44 (CH INNOCARD, ICP BIA),
//  45 (NL CCV, VX810),
//  46 (SE/DK/NO/FI/BALTIC STATES/PL, BAXI),
//  47 (RESERVED FOR FUTURE USE),
//  48 (CH ICP BIA ),
//  49 (US VERIFONE PCCHARGE THIN CLIENT),
//  50 (AT HOBEX),
//  51 (CZ WNSP INGENICO),
//  52 (CH ADUNO/PAYTEC PRIMUS),
//  53 (CH INGENICO CRP 3.12),
//  54 (FI POINT),
//  55 (DE VERIFONE H5000),
//  56 (DE/UK VERIFONE VX825),
//  57 (DE INGENICO IPP480),
//  58 (DE INGENICO ICT250),
//  59 (AT PAYLIFE VERIFONE VX825),
//  60 (CH INNOCARD INGENICO IPP480),
//  61 (FR/ES/UK INGENICO AXIS/C3 WITH CALLBACKS),
//  62 (BE ATOS NVIC 1.07/13),
//  63 (NL ATOS NVIC 1.07/13),
//  64 (LU ATOS NVIC 1.07/13),
//  65 (NL BANKSYS YOMANI 1.06-NL),
//  66 (SK WNSP INGENICO),
//  67 (RO WNSP INGENICO),
//  68 (CH VIBBEK),
//  69 (CA/US AJB FIPAY),
//  70 (UK GLOBAL PAYMENTS EPOS INTERFACE) 
//  71 (AU PC-EFTPOS)
//  72 (CH SIX CCREDIT - OPI)
//  73 (HK PAX INTERFACE FOR CREDIT AND CUP / EPS CARDS)
//  74 (SE VERIFONE/POINT)
//  75 (JP PANASONIC)
//  76 (DE VERIFONE MX925)
//  77 (TH Hypercom VTINT protocol Thailand)
//  78 (CH INGENICO B&S, CRP 3.13B)
//  79 (CH INGENICO CCV, CRP 3.13B)
//  80 (CH Loyalty Giftcard 1.5.4)
//  81 (SG S60-T, PAX) 
//  82 (MY S60-T, PAX)
//  83 (UK Verifone, Premier Connect Protocol)
//  84 (AT Hobex ZVT)
//  87 (IT INGENICO, Protocol 17 Version 3.1)
//  88 (PT Adyen Interface)
//  89 (AT PAYLIFE VERIFONE VX825, DEBIT CARD STORNO),
//  91 (DE/UK VERIFONE VX825, WITH TIP FUNCTION),
//  92 (HK SPECTRA TECHNOLOGIES PR608)
//  93 (DK INGENICO IPP 350)
//  94 (UK/IE FIS TransaxEFT Vx820)
//  96 (UK/IE ELAVON Smartlink iCT250)
//  97 (EU CCV Pad)
//  98 (MO/MY Global Payment Vx690)
//  103 (Ingenico Telium Tetra Serie CH CRP 3.13B )
//  104 (Ingenico Telium Tetra Serie DE / ZVT/A32.de)
//  105 (Landi A8, ChinaUMS)
//  113 (CCV NL Vx820 O.P.I)
//  117 (BBL ECR - Virtual Terminal Interface)
// 999 (PEPPER IMPLEMENTATION MOCK)
// Values for output, representing the card type that will be assigned:
// -------------------------------------Card not recognized-------------------------------------
// 0 = UNKNOWN CARD
// -------------------------------------International cards-------------------------------------
// 1 = MAESTRO
// 2 = MASTERCARD
// 3 = JCB
// 4 = DINERS
// 5 = VISA
// 6 = AMEX
// 7 = JAPAN AIRLINES
// 8 = EAPS (Euro Alliance of Payment Schemes)
// 9 = GIROCARD
// 10 = VISA ELECTRON
// 11 = V PAY
// 12 = CUP (China Union Pay)
// 13 = DEBIT
// 14 = VISA Debit
// 15 = Tamoil Card
// 16 = PayPal
// 17 = AliPay
// 18 = WeChat Pay
// 19 = SONECT
// 20 = RUPAY
// 21 = GiftCard POL
// 22 = BLUECODE
// 23 = CUP QR CODE
// 24 = CUP DEBIT
// -------------------------------------Swiss cards-------------------------------------
// 100 = MAESTRO-CH
// 101 = M-CARD
// 102 = POSTCARD
// 103 = BERN CARD
// 104 = ZUERI-CARD
// 105 = BASLER-CARD
// 106 = CASH
// 107 = PHARMACARD (former COVERCARD)
// 108 = MOEVENPICK
// 109 = ACCESSARENA
// 110 = SHOPPINGBONUS
// 111 = MYONE
// 112 = ORANGE
// 113 = SUNRISE
// 114 = SWISSCOM
// 115 = ACCARDA
// 116 = COOP SUPERCARD PLUS MASTERCARD
// 117 = POWERCARD
// 118 = WIR
// 119 = IKEA-CARD
// 120 = REKA
// 121 = STAEDTECARD
// 122 = JGIFT (JELMOLI)
// 123 = ADUNO KUNDENKARTE
// 124 = SWISS BONUS CARD
// 125 = ABO CARD (Aduno)
// 126 = MEDIAMARKT CARD
// 127 = YALLO
// 128 = COOP MOBILE
// 129 = PHONECARD
// 130 = ORANGECLICK.CH
// 131 = M-BUDGET MOBILE
// 132 = M-BUDGET MASTERCARD
// 133 = GIFTCARD (SIX)
// 134 = COOP SUPERCARD PLUS VISA
// 135 = LOKALKARTE (READ OVER USB-PORT)
// 136 = SWISSCADEAU
// 137 = JELMOLI PAYCARD
// 138 = BARRACUDA TEMPCARD
// 139 = CONFORAMA CARD
// 140 = BONGENIE/GRIEDER CARD
// 141 = GIFT CARD BILLAG
// 142 = ADUNO TEST
// 143 = POSTFINANCE MOBILE COUPONING
// 144 = POSTFINANCE MOBILE PAYMENT
// 145 = KUNDENKARTE OFAC
// 146 = REINVEST SELECT COMPANY CARD
// 147 = KUNDENKARTE PKZ
// 148 = JUBIN CARD
// 149 = INNOCARD KUNDENKARTE
// 150 = ABO CARD
// 151 = SWISS HAPPY CARD
// 153 = DATA STANDARDS
// 154 = SWISSBONUS+
// 155 = (SWISS BONUS CARD) deprecated, type 124 is used instead
// 156 = GE MONEY BANK
// 157 = BARRACUDA TEMPCARD
// 158 = MERGER CARD
// 159 = MERGER CARD 2
// 160 = BONCARD POINTS (CUSTOMER LOYALTY SYSTEM)
// 161 = BONCARD PAY (BONCARD VOUCHER)
// 162 = FNAC-CEMBRA
// 163 = COMMIT CARD COMPANY CARD
// 164 = KUNDENKARTE LOEB
// 165 = EXLIBRIS GESCHENKKARTE
// 166 = SCHILD GESCHENKKARTE
// 167 = SIHLCITY GESCHENKKARTE
// 168 = PLUSPUNKT GESCHENKKARTE
// 169 = SWISSTOP GESCHENKKARTE
// 170 = CITYCARD LUZERN
// 171 = BILLPAY
// 172 = LUNCH-CHECK CARD
// 173 = CCV GIFT CARD
// 174 = COOP EKZ
// 175 = SBB GESCHENKKARTE / CARTE CADEAU CFF
// 176 = CO2NEUTRAL PAY
// 177 = PAYSYS PAY
// 178 = REKA LUNCH CARD
// 179 = BONCARD TEST (DIRECT)
// 180 = REKA RAIL
// 181 = REKA E-COMMERCE
// 182 = JPAYC (JELMOLI)
// 183 = GIFTCARD (Loyalty)
// 184 = TWINT (Mobile Payment no physical card)
// 185 = Paymit (Mobile Payment no physical card)
// 186 = GROUPCARD (Loyalty Gift Card AG)
// 187 = SUPERCARD Mastercard
// 188 = SUPERCARD Visa Credit
// 189 = SUPERCARD Visa Prepaid
// 199 = E-LOAD (mobile number, no physical card)
// 2000 = SWISS PAY
// 2001 = SWISS PAY 2
//-------------------------------------Austrian cards-------------------------------------
// 200 = QUICK
// 201 = MAESTRO_AT
// 203 = IKEA FAMILY PAYCARD
// 204 = FLOTTENKARTE (TYP 4)
// 217 = AUSTRIA CARD
// 218 = OEBB VORTEILSKARTE
// 219 = SHOPIN-KARTE
// 220 = TLAPA-KARTE
// 221 = Inside Dornbirn
// 222 = VALUEMASTER
//-------------------------------------Slovenian cards-------------------------------------
// 300 = ACTIVA NAVADNA
// 301 = BA DEBIT CARD
// 302 = KARANTA
//-------------------------------------German cards-------------------------------------
// 401 = MAESTRO_DE
// 402 = GELDKARTE
// 403 = IKEA FAMILY PLUS
// 404 = BSW
// 405 = GOBANANAS
// 409 = LASTSCHRIFT
// 410 = MAESTRO ELV
// 411 = MAESTRO OLV
// 412 = EC-CASH
// 413 = EUROELV
// 414 = ANNICARD
// 420 = HORNBACH PROFI
// 421 = HORNBACH WELT
// 422 = HORNBACH GUTSCHEINKARTE
// 431 = PREMIERE FLEX
// 438 = BREUNINGER
// 440 = INTERCARD GUTSCHEINKARTE
// 441 = T-D1 XTRA
// 442 = COMFORT CARD
// 452 = PAYSAFECARD
// 479 = VODAFONE CALLYA
// 480 = E-PLUS FREE&EASY
// 481 = O2 LOOP	
// 482 = TEST PINPRINTING
// 483 = GESCHENKGUTSCHEIN
// 484 = SCHLECKER PREPAID
// 485 = VITALSANA
// 486 = SCHLECKER SMOBIL
// 487 = SCHLECKER ZUSATZPRODUKTE
// 488 = FONIC
// 489 = BILDMOBIL
// 490 = BLAU.DE
// 491 = CONGSTAR
// 492 = LEBARA MOBILE
// 493 = SIMYO
// 494 = LYCAMOBILE
// 495 = GT MOBILE
// 496 = AVS GUTSCHEINKARTE
// 497 = ALNATURA GUTSCHEINKARTE
//-------------------------------------Belgian cards-----------------------------------
// 500 = BANCONTACT/MISTER CASH
// 501 = PROTON
// 502 = AURORA
// 503 = KADOZ
// 504 = EDENRED
// 505 = SODEXO EPASS
// 506 = MONIZZE
//-------------------------------------US cards----------------------------------------
// 600 = DISCOVER
// 601 = CARTE BLANCHE
// 602 = EN ROUTE (now merged with DINER'S CLUB INTERNATIONAL)
//-------------------------------------Dutch cards-------------------------------------
// 700 = BANCONTACT/MISTER CASH
// 701 = GIRO
// 702 = ELECTRONISCHE POST WISSEL (EPWS)
// 704 = COMFORT CARD (CMFC)
// 705 = PRIME LINE (PRLI)
// 706 = BASIC CARD (BACA)
// 707 = SCHOENENBON
// 708 = FASHION CARD
// 709 = CCV CARD
// 710 = YOURGIFT H&G
// 711 = YOURGIFT G&M
// 712 = YOURGIFT M&M
// 713 = YOURGIFT B&H
// 714 = YOURGIFT H&T
// 715 = YOURGIFT F&F
// 716 = YOURGIFT W&D
// 717 = YOURGIFT G&C
// 718 = HORNBACH PROFICARD NL
// 719 = CHIPKNIP
// 720 = JEWEL CARD
// 730 = INTERSOLVE
//-------------------------------------Italian cards----------------------------------
// 801 = PAGOBANCOMAT
// 802 = KEY CLIENT
// 803 = CARTASI
// 804 = AGOS
// 805 = FINDOMESTIC
// 806 = FINDOMESTIC GIFT
// 807 = FINDOMESTIC AURA
// 808 = FINDOMESTIC NOVA
// 809 = AGOS ITA
// 810 = AGOS ALSOLIA
// 811 = FIDITALIA
// 812 = FIDITALIA PROMOZIONI
// 813 = MONETA VISA
// 814 = MONETA MC
// 815 = CREDIRAMA
// 816 = ACCORD
// 817 = CLESSIDRA
// 818 = LINEA
// 819 = TOYS
// 820 = METRO
// 821 = BANCO DI BRESCIA
//-------------------------------------French cards-----------------------------------
// 900 = CARTE BLEUE (DOMESTIC VISA)
// 901 = ACCORD
// 902 = ALPHYRA (Payzone)
// 903 = CAFETERIA CASINO
// 904 = AUCHAN TELECOM
// 905 = PNF CREDIT MUTUEL
// 906 = EMPLOYE ED
// 907 = EVEREST
// 908 = COFINOGA
// 909 = CHEQUE
// 910 = CALEDINIENNE DE SERVICE BANCAIRE
// 911 = AURORE
// 912 = DATAVANTAGE (GIFTCARD) 
// 913 = EDEL, LECLERC
// 914 = ELF (Total, Parking)
// 915 = PLURIEL - FRANFINANCE
// 916 = FNAC
// 917 = FINAREF
// 918 = GAV (Gestion de Avoirs) 
// 919 = GLF (Galerie Lafayette) 
// 920 = MGL (Magellan)
// 921 = MULTISSIME
// 922 = OPTALION (credit Lyonnais)
// 923 = PASS
// 924 = PGS (Pegasus)
// 925 = PME (Moneo - Porte Monnaie Electronique)
// 926 = PROMOSTIM (Record Card)
// 927 = PROSODIE
// 928 = SOFINCO
// 929 = SODELEM (carte petrole)
// 930 = SMILES
// 931 = CETELEM
// 932 = MODEUS
// 933 = CARTE PINAULT PRINTEMPS REDOUTE
// 935 = CarteBancaire
//-------------------------------------Luxembourgian cards----------------------------------
// 1000 = BANCOMAT LUXEMBOURG
// 1001 = AURORA LUXEMBOURG
// 1002 = HORNBACH PROFICARD LUX
//-------------------------------------Scandinavian cards-----------------------------------
// 1100 = DANKORT
// 1101 = VISA DANMARK
// 1102 = MASTERCARD DK
// 1103 = MAESTRO DK
// 1104 = DINERS DK
// 1105 = AMEX DK
// 1106 = PBS KJEDEKORT / GAVEKORT
// 1107 = PBS CENTERKORT
// 1108 = FORBRUGSFORENINGEN
// 1109 = IKANO FINANS
// 1110 = SPAREXPRESS
// 1111 = BANKAXEPT
// 1112 = PANKKIKORTTI (BANK CARD)
// 1113 = MASTERCARD PANKKIKORTTI YHDISTELMÄKORTTI 
// 1114 = VISA PANKKIKORTTI YHDISTELMÄKORTTI
// 1115 = OPTION
// 1116 = PROFFS KORT
// 1117 = PROJEKTVÄRLDSKORT
//-------------------------------------Japanese cards --------------------------------------------
// 1200 = SAISON CARD
// 1201 = J-DEBIT
// 1202 = JCB GIFT CARD
// 1203 = VISA GIFT CARD
// 1204 = DINERS GIFT CARD
//-------------------------------------United Kingdom cards -------------------------------------------------
// 1300 = DELTA
// 1301 = Maestro UK
//-------------------------------------Hong Kong cards -------------------------------------------------
// 1400 = EPS (ELECTRONIC PAYMENTS SERVICES)
// 1401 = OCTOPUS CARD (for public transports)
//-------------------------------------Malaysia/Macao cards -------------------------------------------------
// 1601 = BPI CARDS
// 1602 = BANCNET
//-------------------------------------Canadian cards -------------------------------------------------
// 1700 = INTERAC CARD
//-------------------------------------Other cards -------------------------------------------------
// 5000 = AirPlus Company
//
//------------------------------------- Values for cardname -------------------------------------
// A card name can be set to overwrite missing or an existing, but wrong, card label. 
// This value is optional, and disregarded, if not set.
//-----------------------------------------------------------------------------------------------
//
// **********************************************************************************************************************************************
// Terminal type 0 (CH ASCOM):  /Extension for PaySys
// The card recognition is based on the ISSUER NUMBER
// **********************************************************************************************************************************************
// remove comments from lines starting by a 0 if ASCOM Terminal or ASCOM EFT simulator is used
//
// MAESTRO ----------------------------
//0.[9]=1;
//0.[09]=1;
// MASTERCARD -------------------------
//0.[4]=2;
//0.[04]=2;
// DINERS -----------------------------
//0.[6]=4;
//0.[06]=4;
// VISA -------------------------------
//0.[7]=5;
//0.[07]=5;
// AMEX -------------------------------
//0.[5]=6;
//0.[05]=6;
// MAESTRO CH -------------------------
//0.[2]=100;
//0.[02]=100;
// M-CARD -----------------------------
//0.[1]=101;
//0.[01]=101;
// POSTCARD ---------------------------
//0.[3]=102;
//0.[03]=102;
// BERN CARD --------------------------
//0.[89]=103;
// ZUERI CARD -------------------------
//0.[90]=104;
//
//if PaySys is used in Ascom Modus
//
// MAESTRO ----------------------------
0.[7]=1;
0.[07]=1;
// MASTERCARD -------------------------
0.[2]=2;
0.[02]=2;
// JCB --------------------------------
0.[6]=3;
0.[06]=3;
// DINERS -----------------------------
0.[43]=4;
// VISA -------------------------------
0.[5]=5;
0.[05]=5;
// AMEX -------------------------------
0.[40]=6;
// MAESTRO CH -------------------------
0.[1]=100;
0.[01]=100;
// M-CARD -----------------------------
0.[22]=101;
// POSTCARD ---------------------------
0.[21]=102;
// STAEDTE CARD -----------------------
0.[41]=103;
0.[42]=103;
0.[45]=103;
// CASH -------------------------------
0.[8]=106;
0.[08]=106;
// PHARMACARD --------------------------
0.[55]=107;
// MOEVENPICK -------------------------
0.[46]=108;
// MOEVENPICK -------------------------
0.[81]=110;
// MYONE ------------------------------
0.[84]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
0.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
0.[26]=134;
// BONCARD
0.[66]=160,BONARD POINTS;
0.[67]=161,BONCARD PAY;
// LUNCH-CHECK
0.[37]=172;
// CCV PAY
0.[35]=173;
// COOP EKZ
0.[37]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
0.[60]=175;
// **********************************************************************************************************************************************
// Terminal type 1 (CH SIX PAYMENT SERVICES  / AID Brandtabelle 03.08.2010):
// The card recognition is based on the ISSUER NUMBER
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
1.[7]=1;
1.[07]=1;
// MASTERCARD -------------------------
1.[2]=2;
1.[02]=2;
// M-Budget MasterCard
1.[25]=2;
// JCB --------------------------------
1.[6]=3;
1.[06]=3;
// DINERS -----------------------------
1.[43]=4;
// VISA / VISA ELECTRON ---------------
1.[5]=5;
1.[05]=5;
// V PAY ------------------------------
1.[96]=11,V PAY;
// AMEX -------------------------------
1.[40]=6;
// BANCOMAT CETREL LUXEMBOURG ----------
1.[30]=1000,BANCOMAT LUXEMOURG;
// AURORA LUXEMBOURG -------------------
1.[31]=1001,AURORA LUXEMBOURG;
// PHARMACARD -------------------------
1.[55]=107,PHARMACARD;
// MAESTRO CH -------------------------
1.[1]=100;
1.[01]=100;
// CUP ---------------------------------
1.[9]=12,CHINA UNION PAY;
1.[09]=12,CHINA UNION PAY;
// M-CARD -----------------------------
1.[22]=101;
// POSTCARD ---------------------------
1.[21]=102;
// BERN CARD --------------------------
1.[42]=103;
// ZUERI CARD, IKEA-CARD OR STAEDTECARD-
//1.[41]=104,ZUERI CARD;
1.[41]=119,IKEA-CARD   ; 
1.[+6004520306]=119,IKEA-CARD   ;
1.[91]=119,IKEA-CARD   ;
//1.[41]=121,STAEDTECARD;
1.[+60045203]=121,STAEDTECARD;
// BASLER CARD ------------------------
1.[45]=105;
// CASH -------------------------------
1.[8]=106;
1.[08]=106;
// JELMOLI BONUS SHOPPING CARD --------
1.[81]=110;
// JELMOLI PAYCARD --------------------
1.[a0000001574462]=137;
// MOEVENPICK -------------------------
1.[46]=108;
// MANOR CARD myONE -------------------
1.[84]=111;
// ORANGE -----------------------------
1.[75]=112;
// SUNRISE ----------------------------
1.[76]=113;
// SWISSCOM ---------------------------
1.[77]=114;
// COOP SUPERCARD PLUS MASTERCARD-------
1.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
1.[26]=134;
// POWERCARD --------------------------
1.[87]=117;
1.[47]=117;
// GIFT CARD ACCARDA -------------------
1.[88]=115;
// WIR --------------------------------
1.[50]=118; 
// REKA  -------------------------------
1.[90]=120;
// REKA RAIL ---------------------------
1.[45]=180;
// REKA LUNCH ---------------------------
1.[78]=178;
// JGIFT (JELMOLI)---------------------
1.[72]=122,J-Geschenkkarte;
// COMMTRAIN/ADUNO CARDS --------------
// MEDIAMARKT -------------------------
1.[89]=126,Mediamarkt Card;
// YALLO ------------------------------
1.[73]=127,Yallo;
// COOP MOBILE ------------------------
1.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
1.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
1.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
1.[82]=131,M-Budget Mobile;
// GIFTCARD (SIX)-----------------------
1.[97]=133,GiftCard;
// BONCARD
1.[66]=160,BONARD POINTS;
1.[67]=161,BONCARD PAY;
// LUNCH-CHECK
1.[37]=172;
// CCV PAY
1.[35]=173;
// COOP EKZ
1.[37]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
1.[60]=175;
//
// **********************************************************************************************************************************************
// Terminal type 2 (CH EPSYS):
// The card recognition is based on the ISSUER NUMBER
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
2.[7]=1;
2.[07]=1;
// MASTERCARD -------------------------
2.[2]=2;
2.[02]=2;
// M-Budget MasterCard
2.[25]=2;
// JCB --------------------------------
2.[6]=3;
2.[06]=3;
// DINERS -----------------------------
2.[43]=4;
// VISA / VISA ELECTRON ---------------
2.[5]=5;
2.[05]=5;
// V PAY ------------------------------
2.[96]=11,V PAY;
// AMEX -------------------------------
2.[40]=6;
// BANCOMAT LUXEMBOURG -----------------
2.[30]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG -------------------
2.[31]=1001,AURORA LUXEMBOURG;
// PHARMACARD -------------------------
2.[55]=107,PHARMACARD;
// MAESTRO CH -------------------------
2.[1]=100;
2.[01]=100;
// CUP ---------------------------------
2.[9]=12,CHINA UNION PAY;
2.[09]=12,CHINA UNION PAY;
// M-CARD -----------------------------
2.[22]=101;
// POSTCARD ---------------------------
2.[21]=102;
// BERN CARD --------------------------
2.[42]=103;
// ZUERI CARD, IKEA-CARD OR STAEDTECARD-
//2.[41]=104,ZUERI CARD;
2.[41]=119,IKEA-CARD   ; 
2.[+6004520306]=119,IKEA-CARD   ;
2.[91]=119,IKEA-CARD   ;
//2.[41]=121,STAEDTECARD;
2.[+60045203]=121,STAEDTECARD;
// BASLER CARD ------------------------
2.[45]=105;
// CASH -------------------------------
2.[8]=106;
2.[08]=106;
// JELMOLI BONUS SHOPPING CARD --------
2.[81]=110;
// JELMOLI PAYCARD --------------------
2.[a0000001574462]=137;
// MOEVENPICK -------------------------
2.[46]=108;
// MANOR CARD myONE -------------------
2.[84]=111;
// ORANGE -----------------------------
2.[75]=112;
// SUNRISE ----------------------------
2.[76]=113;
// SWISSCOM ---------------------------
2.[77]=114;
// COOP SUPERCARD PLUS MASTERCARD-------
2.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
2.[26]=134;
// POWERCARD --------------------------
2.[87]=117;
2.[47]=117;
// GIFT CARD ACCARDA -------------------
2.[88]=115;
// WIR --------------------------------
2.[50]=118; 
// REKA  -------------------------------
2.[90]=120;
// REKA RAIL ---------------------------
2.[45]=180;
// REKA LUNCH ---------------------------
2.[78]=178;
// JGIFT (JELMOLI)---------------------
2.[72]=122,J-Geschenkkarte;
// COMMTRAIN/ADUNO CARDS --------------
// MEDIAMARKT -------------------------
2.[89]=126,Mediamarkt Card;
// YALLO ------------------------------
2.[73]=127,Yallo;
// COOP MOBILE ------------------------
2.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
2.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
2.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
2.[82]=131,M-Budget Mobile;
// GIFTCARD (SIX)-----------------------
2.[97]=133,GiftCard;
// BONCARD
2.[66]=160,BONARD POINTS;
2.[67]=161,BONCARD PAY;
// LUNCH-CHECK
2.[37]=172;
// CCV PAY
2.[35]=173;
// COOP EKZ
2.[37]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
2.[60]=175;
// GROUPCARD (Loyalty Gift Card AG)------
2. [+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 3 (CH CASH):
// The card recognition is based on the ISSUER NUMBER
// (The only possible card that can be used, is the CASH Card)
// **********************************************************************************************************************************************
// CASH -------------------------------
3.[1]=106;
3.[01]=106;
// GROUPCARD (Loyalty Gift Card AG)------
3. [+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 4 (CH CCREDIT):
// The card recognition is based on the ISSUER NUMBER and Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
4.[7]=1;
4.[07]=1;
// MASTERCARD -------------------------
4.[2]=2;
4.[02]=2;
4.[+a0000000041010]=2;
// M-Budget MasterCard
4.[25]=2;
// JCB --------------------------------
4.[6]=3;
4.[06]=3;
// DINERS -----------------------------
4.[43]=4;
// VISA / VISA ELECTRON ---------------
4.[5]=5;
4.[05]=5;
4.[+a0000000031010]=5;
4.[+a0000000032010]=10;
// V PAY -------------------------------
4.[96]=11,V PAY;
4.[+a0000000032020]=11,V PAY;
4.[a0000001574452]=11,V PAY;
// CUP (China UnionPay)----------------
4.[9]=12,CUP;
4.[09]=12,CUP;
4.[a000000333010101]=12,CHINA UNION PAY;
4.[a000000333010102]=12,CHINA UNION PAY;
// AMEX -------------------------------
4.[40]=6;
4.[a000000025010402]=6;
4.[a0000000291010]=6;
// MAESTRO CH -------------------------
4.[1]=100;
4.[01]=100;
4.[a0000000031030]=100;
// M-CARD -----------------------------
4.[22]=101;
// POSTCARD ---------------------------
4.[21]=102;
// STAEDTECARD -(choose your card here)-
//4.[41]=103,BERN CARD;
4.[41]=104,ZUERI-CARD;
//4.[41]=105,BASLER-CARD;
// JELMOLI SHOPPING BONUS CARD --------
4.[81]=110;
4.[a00000000300037561]=110;
// JELMOLI PAYCARD --------------------
4.[a0000001574462]=137;
// MANOR CARD myONE -------------------
4.[84]=111;
// ORANGE -----------------------------
4.[75]=112;
// Sunrise ----------------------------
4.[76]=113;
// SWISSCOM ---------------------------
4.[77]=114;
// COOP SUPERCARD PLUS MASTERCARD-------
4.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
4.[26]=134;
// WIR --------------------------------
4.[50]=118;
// PHARMACARD -------------------------
4.[55]=107,PHARMACARD;
// SWISS BONUS CARD -------------------
4.[56]=124,SWISS BONUS CARD;
// REKA -------------------------------
4.[90]=120;
4.[+a000000157010E]=120;
4.[d7560000010101]=120;
4.[a0000001880101]=120;
4.[a0000001574464]=120;
4.[d7560000011111]=120;
// REKA RAIL --------------------------
4.[45]=180;
4.[d7560000011112]=180;
4.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
4.[78]=178;
4.[d7560000011113]=178;
4.[d7560000010102]=178;
4.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
4.[a000000157448C]=181;
// ABO CARD ---------------------------
4.[95]=125,ABO Card;
// YALLO ------------------------------
4.[73]=127,Yallo;
// COOP MOBILE ------------------------
4.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
4.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
4.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
4.[82]=131,M-Budget Mobile;
// POWERCARD (MF Group) -----------------
4.[87]=117;
4.[47]=117;
4.[a00000015701]=117;
4.[a0000001574476]=117;
4.[a0000001574478]=117;
4.[a0000001574479]=117;
// SWISSCADEAU --------------------------
4.[a000000157445A]=136,SWISSCADEAU;
// KUNDENKARTE BONGENIE/GRIEDER ---------
4.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
4.[a0000001574455]=160,BONARD POINTS;
4.[a000000157445B]=161,BONCARD PAY;
4.[66]=160,BONARD POINTS;
4.[67]=161,BONCARD PAY;
// LUNCH-CHECK --------------------------
4.[37]=172;
4.[a000000157447d]=172;
4.[d7560001150001]=172;
// CCV PAY ------------------------------
4.[35]=173;
4.[a000000157447b]=173;
// COOP EKZ -----------------------------
4.[37]=174;
4.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
4.[60]=175;
4.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
4.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
4.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
4.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
4. [+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 5 (AT FIRST DATA):
// The card recognition is based on the CARD NAME, given by the interface.
// (As default field m_name or, in the case of debit cards, the 3 first characters of p_text)
// **********************************************************************************************************************************************
// MAESTRO PIN based-------------------
5.[edc]=1;
// MASTERCARD PIN based----------------
5.[KMC]=2;
// MASTERCARD signature based----------
5.[+mastercard]=2;
5.[+mc ]=2;
// JCB PIN based-----------------------
5.[KJB]=3;
// JCB signature based-----------------
5.[+jcb]=3;
// DINERS PIN based--------------------
5.[KDI]=4;
// DINERS signature based--------------
5.[+diners]=4;
// VISA ELECTRON-----------------------
5.[+visa elec]=10;
// V PAY-------------------------------
5.[+v pay]=11;
5.[KVP]=11;
// VISA PIN based----------------------
5.[KVI]=5;
// VISA signature based----------------
5.[+visa]=5;
5.[citibank visa]=5;
5.[hanseatic bank]=5;
5.[anzsmartpay]=5;
5.[ykb worldcard]=5;
// AMEX PIN based----------------------
5.[KAM]=6;
// AMEX signature based----------------
5.[+american]=6;
5.[+amex]=6;
// QUICK PIN based---------------------
5.[QCK]=200;
// BANKOMAT (MAESTRO AT) PIN based-----
5.[B-K]=201;
// BANKOMAT PIN based (not guaranteed)-
5.[BKN]=201;
5.[bkn]=201;
// JAPAN AIRLINES ---------------------
5.[+japan]=7;
// Euro Alliance of Payment Schemes----
5.[+eaps]=8;
// GIROCARD----------------------------
5.[+giro]=9;
// CUP (China UnionPay)----------------
5.[+cup]=12;
5.[+KCU]=12;
5.[+china]=12;
// FLOTTENKARTE (TYP 4)
5.[TYP4]=204,Flottenkarte;
// LOKALE KARTE (TYP3)
5.[TYP3]=205;
//
// **********************************************************************************************************************************************
// Terminal type 6 (SI BANKART):
// The card recognition is primarily based on the CARD NAME and secondly on the CARD NUMBER PREFIX.
// **********************************************************************************************************************************************
// ACTIVA NAVADNA ---------------------
6.[+activa]=300;
6.[+0059]=300;
// BA DEBIT CARD ----------------------
6.[+ba ]=301;
6.[+ba/]=301;
6.[nlb ba]=301;
6.[+676226]=301;
6.[+676268]=301;
6.[+676270]=301;
6.[+676324]=301;
6.[+676325]=301;
6.[+676326]=301;
6.[+676327]=301;
6.[+676385]=301;
6.[+676412]=301;
6.[+677137]=301;
6.[+677141]=301;
6.[+677143]=301;
6.[+677152]=301;
6.[+677181]=301;
6.[+677173]=301;
6.[+677192]=301;
6.[+677195]=301;
// KARANTA ----------------------------
6.[karanta]=302;
6.[nlb karanta]=302;
6.[+591,-59184,-59185]=302;
// DINERS -----------------------------
6.[+diners]=4;
6.[+300]=4;
6.[+301]=4;
6.[+302]=4;
6.[+303]=4;
6.[+304]=4;
6.[+305]=4;
6.[+3095]=4;
6.[+3095]=4;
6.[+3096]=4;
6.[+3097]=4;
6.[+3098]=4;
6.[+3099]=4;
6.[+36]=4;
6.[+38]=4;
6.[+39]=4;
6.[+60110]=4;
6.[+60112]=4;
6.[+60113]=4;
6.[+60114]=4;
6.[+601174]=4;
6.[+601177]=4;
6.[+601178]=4;
6.[+601179]=4;
6.[+601186]=4;
6.[+601187]=4;
6.[+601188]=4;
6.[+601189]=4;
6.[+60119]=4;
6.[+644]=4;
6.[+645]=4;
6.[+646]=4;
6.[+647]=4;
6.[+648]=4;
6.[+649]=4;
6.[+65,-6541,-6556]=4; 
// MASTERCARD -------------------------
6.[+eurocard]=2;
6.[+master]=2;
6.[+500235]=2;
6.[+500766]=2;
6.[+501012]=2;
6.[+502029]=2;
6.[+503615]=2;
6.[+504507]=2;
6.[+504834]=2;
6.[+504837]=2;
6.[+5049]=2;
6.[+51]=2;
6.[+52]=2;
6.[+53]=2;
6.[+54,-545250]=2;
6.[+55]=2;
6.[+222]=2;
6.[+223]=2;
6.[+224]=2;
6.[+225]=2;
6.[+226]=2;
6.[+227]=2;
6.[+228]=2;
6.[+229]=2;
6.[+230]=2;
6.[+231]=2;
6.[+232]=2;
6.[+233]=2;
6.[+234]=2;
6.[+235]=2;
6.[+236]=2;
6.[+237]=2;
6.[+238]=2;
6.[+239]=2;
6.[+240]=2;
6.[+241]=2;
6.[+242]=2;
6.[+243]=2;
6.[+244]=2;
6.[+245]=2;
6.[+246]=2;
6.[+247]=2;
6.[+248]=2;
6.[+249]=2;
6.[+250]=2;
6.[+251]=2;
6.[+252]=2;
6.[+253]=2;
6.[+254]=2;
6.[+255]=2;
6.[+256]=2;
6.[+257]=2;
6.[+258]=2;
6.[+259]=2;
6.[+261]=2;
6.[+262]=2;
6.[+263]=2;
6.[+264]=2;
6.[+265]=2;
6.[+266]=2;
6.[+267]=2;
6.[+268]=2;
6.[+269]=2;
6.[+270]=2;
6.[+271]=2;
6.[+272]=2;
// MAESTRO ----------------------------
6.[+maestro]=1;
6.[nlb maestro]=1;
6.[+50]=1;
6.[+54250]=1;
6.[+56]=1;
6.[+57]=1;
6.[+58]=1;
6.[+59]=1;
6.[+60]=1;
6.[+61]=1;
6.[+62]=1;
6.[+63]=1;
6.[+64]=1;
6.[+66]=1;
6.[+67]=1;
6.[+68]=1;
6.[+69]=1; 
// VISA ELECTRON -----------------------
6.[+400115]=10;
6.[+400837]=10;
6.[+400838]=10;
6.[+400839]=10;
6.[+412921]=10;
6.[+412922]=10;
6.[+412923]=10;
6.[+417935]=10;
6.[+419740]=10;
6.[+419741]=10;
6.[+419773]=10;
6.[+419774]=10;
6.[+419775]=10;
6.[+424519]=10;
6.[+424962]=10;
6.[+424963]=10;
6.[+437860]=10;
6.[+444000]=10;
6.[+459472]=10;
6.[+484406]=10;
6.[+484407]=10;
6.[+484408]=10;
6.[+484409]=10;
6.[+484410]=10;
6.[+484411]=10;
6.[+484413]=10;
6.[+484414]=10;
6.[+484418]=10;
6.[+484419]=10;
6.[+484420]=10;
6.[+484421]=10;
6.[+484422]=10;
6.[+484423]=10;
6.[+484424]=10;
6.[+484425]=10;
6.[+484426]=10;
6.[+484428]=10;
6.[+484429]=10;
6.[+48443]=10;
6.[+48444]=10;
6.[+484450]=10;
6.[+484451]=10;
6.[+484452]=10;
6.[+484453]=10;
6.[+484454]=10;
6.[+484455]=10;
6.[+49173]=10;
6.[+49174]=10;
6.[+49175]=10; 
// VISA -------------------------------
6.[+placilo s]=5;
6.[+visa]=5;
6.[+4]=5;
// AMEX -------------------------------
6.[+centurion]=6;
6.[+amex]=6;
6.[+34]=6;
6.[+37]=6;
// JCB --------------------------------
6.[+japan]=3;
6.[+jcb]=3;
6.[+35]=3; 
//
// **********************************************************************************************************************************************
// Terminal type 7 (CH SIX PAYMENT SERVICES MPD): 
// The card recognition is based on AID, ISSUER NUMBER, Card Number and Card Name
// **********************************************************************************************************************************************
// where Issuer-Nbr exist do not use AID, rely on Issuer-Nbr from terminal
// E-LOAD (mobile number, no physical card)
7.[+076]=199;
7.[+077]=199;
7.[+078]=199;
7.[+079]=199;
// MAESTRO ----------------------------
7.[7]=1;
7.[07]=1;
7.[+a0000000043060]=1;
// MASTERCARD -------------------------
7.[2]=2;
7.[02]=2;
7.[+a0000000041010]=2;
// M-Budget MasterCard
7.[25]=132;
7.[D7560000300101]=132;
7.[A000000157444B]=132;
7.[Cumulus MC]=132;
// JCB --------------------------------
7.[06]=3;
7.[6]=3;
7.[+a0000000651010]=3;
// DINERS -----------------------------
7.[43]=4;
7.[a0000001570060]=4,DINERS;
7.[+a0000001523010]=4,DINERS;
// VISA / VISA ELECTRON ---------------
7.[5]=5;
7.[05]=5;
7.[+a0000000031010]=5;
7.[+a0000001570030]=5;
7.[+a0000000032010]=10;
// V PAY ------------------------------
7.[96]=11,V PAY;
7.[a0000000032020]=11,V PAY; 
7.[a0000001574452]=11,V PAY;
7.[V PAY]=11,V PAY;
// AMEX -------------------------------
7.[40]=6;
7.[a000000025010402]=6;
7.[a0000000291010]=6;
7.[+a00000002501]=6;
7.[a0000001570010]=6, AMERICAN EXPRESS;
// BANCOMAT LUXEMBOURG ----------------
7.[30]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ------------------
7.[31]=1001,AURORA LUXEMBOURG;
// PHARMACARD -------------------------
7.[55]=107,PHARMACARD;
// MAESTRO CH -------------------------
7.[1]=100;
7.[01]=100;
7.[a0000000031030]=100;
// CUP --------------------------------
7.[9]=12,CHINA UNION PAY;
7.[09]=12,CHINA UNION PAY;
7.[a000000333010101]=12,CHINA UNION PAY;
7.[a000000333010102]=12,CHINA UNION PAY;
7.[a000000333010103]=12,CHINA UNION PAY;
// M-CARD -----------------------------
7.[22]=101;
// POSTCARD ---------------------------
7.[21]=102;
7.[a0000001570050]=102,POSTCARD;
7.[a0000001110505]=102,POSTCARD;
7.[a0000001110101]=102,POSTCARD;
7.[a0000001110110]=102,POSTCARD;
7.[a0000001110510]=102,POSTCARD;
7.[a0000001110201]=102,POSTCARD;
7.[a0000001110210]=102,POSTCARD;
// BERN CARD --------------------------
7.[42]=103;
// ZUERI CARD, IKEA-CARD OR STAEDTECARD-
//7.[41]=104,ZUERI CARD;
7.[41]=119,IKEA-CARD   ; 
7.[+6004520306]=119,IKEA-CARD   ;
7.[91]=119,IKEA-CARD   ;
//7.[41]=121,STAEDTECARD;
7.[+60045203]=121,STAEDTECARD; 
// BASLER CARD ------------------------
7.[45]=105;
// CASH -------------------------------
7.[8]=106;
7.[08]=106;
// JELMOLI BONUS SHOPPING CARD --------
7.[81]=110;
// JELMOLI PAYCARD --------------------
7.[a0000001574462]=137;
// MOEVENPICK -------------------------
7.[46]=108;
// MANOR CARD myONE -------------------
7.[84]=111; 
// ORANGE -----------------------------
7.[75]=112;
// SUNRISE ----------------------------
7.[76]=113;
// SWISSCOM ---------------------------
7.[77]=114;
// COOP SUPERCARD PLUS MASTERCARD-------
7.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
7.[26]=134;
7.[a0000000037562]=134;
// SUPERCARD MASTERCARD ----------------
7.[a000000162000040]=187;
7.[SC_MC]=187;
// SUPERCARD VISA CREDIT ---------------
7.[a000000162000041]=188;
7.[SC_Visa]=188;
// SUPERCARD VISA PREPAID --------------
7.[a000000162000042]=189;
7.[SC_Visa_PP]=189;
// POWERCARD --------------------------
7.[87]=117;
7.[47]=117;
7.[a0000001574476]=117;
7.[a0000001574478]=117;
7.[a0000001574479]=117;
// GIFT CARD ACCARDA -------------------
7.[88]=115;
// WIR --------------------------------
7.[50]=118; 
// REKA  -------------------------------
7.[90]=120;
7.[+a000000157010E]=120;
7.[d7560000010101]=120;
7.[a0000001880101]=120;
7.[a0000001574464]=120;
7.[d7560000011111]=120;
// REKA RAIL --------------------------
7.[45]=180;
7.[d7560000011112]=180;
7.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
7.[78]=178;
7.[d7560000011113]=178;
7.[d7560000010102]=178;
7.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
7.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
7.[72]=122,J-Geschenkkarte;
// COMMTRAIN/ADUNO CARDS --------------
// MEDIAMARKT -------------------------
7.[89]=126,Mediamarkt Card;
// YALLO ------------------------------
7.[73]=127,Yallo;
// COOP MOBILE ------------------------
7.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
7.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
7.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
7.[82]=131,M-Budget Mobile;
// GIFTCARD (SIX)-----------------------
7.[97]=133,GiftCard;
// SWISSCADEAU---------------------------
7.[a000000157445A]=136,SWISSCADEAU;
// KUNDENKARTE BONGENIE/GRIEDER ---------
7.[a000000157445E]=140,BonGenie/Grieder Card; 
7.[+636265]=140,BonGenie/Grieder Card;
// BONCARD ------------------------------
7.[66]=160,BONARD POINTS;
7.[67]=161,BONCARD PAY;
// LUNCH-CHECK --------------------------
7.[37]=172;
7.[a000000157447d]=172;
7.[d7560001150001]=172;
// CCV PAY ------------------------------
7.[35]=173;
7.[a000000157447b]=173;
// COOP EKZ -----------------------------
7.[37]=174;
7.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
7.[60]=175;
7.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
7.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
7.[a0000001574482]=177;
// BONCARD TEST (DIREKT) ----------------
7.[a0000001574486]=179;
// KUNDENKARTE LOEB ---------------------
7.[64]=164;
// JELMOLI BONUS SHOPPING CARD ----------
7.[81]=110;
// JPAYC (JELMOLI)-----------------------
7.[65]=182,JPAYC;
// GROUPCARD (Loyalty Gift Card AG)------
7.[+a0000001574480]=186;
// Paymit -------------------------------
7.[a000000157449a]=185,PAYMIT;
7.[a000000162000501]=185,PAYMIT;
7.[a000000162000502-ff]=185,PAYMIT;
7.[28]=185,PAYMIT;
// TWINT -------------------------------
7.[a000000157449e]=184,TWINT;
7.[a000000162000502]=184,TWINT;
7.[29]=184,TWINT;
// BONCARD -------------------------------
7.[a0000001574455]=160,BONCARD POINTS;
7.[a000000157445B]=161,BONCARD PAY;
// ALIPAY ---------------------------
7.[a000000162000503]=17,AliPay;
7.[a00000015744A0]=17,AliPay;
7.[+alipay]=17,AliPay;
7.[28]=17,AliPay;
// WECHAT PAY -----------------------
7.[+wechatpay_pos]=18,WeChat Pay;
7.[+wechatpay]=18,WeChat Pay;
7.[+wechat]=18,WeChat Pay;
// SWISS PAY
7.[a0000008010001]=2000,SWISS PAY;
7.[37]=2000,SWISS PAY;
// SWISS PAY 2
7.[a00000015744BD]=2001,SWISS PAY 2;
7.[37]=2001,SWISS PAY 2;
// FNAC-CEMBRA
7.[A000000157449C]=162,FNAC-CEMBRA;
7.[D7560001160101]=162,FNAC-CEMBRA;
//
// **********************************************************************************************************************************************
// Terminal type 8 (DE ZVT-700-KMS):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
8.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
8.[5]=9,GIROCARD;
// EAPS -------------------------------
8.[7]=8,EAPS;
// Lastschrift ------------------------
8.[9]=409,LASTSCHRIFT;
8.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
8.[11]=10,VISA ELECTRON;
8.[70]=10,VISA ELECTRON;
// V PAY ------------------------------
8.[13]=11,V PAY;
// BSW --------------------------------
8.[40]=404,BSW;
// MAESTRO ----------------------------
8.[46]=1;
// MASTERCARD -------------------------
8.[6]=2;
// DINERS -----------------------------
8.[12]=4;
// VISA -------------------------------
8.[10]=5;
// AMEX -------------------------------
8.[8]=6;
// JCB --------------------------------
8.[14]=3;
// MAESTRO DE / ELV -------------------
8.[2ELV]=410;
// MAESTRO DE / OLV -------------------
8.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
8.[2CASH]=412;
// MAESTRO DE / default ---------------
8.[2]=401;
// GELDKARTE --------------------------
8.[30]=402;
// IKEA FAMILY PLUS -------------------
8.[76]=403;
// HORNBACH Profi ---------------------
8.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
8.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
8.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
8.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
8.[42]=442,Comfort Card;
// Breuninger-- -----------------------
8.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
8.[+414]=414,AnniCard;
8.[+AnniCard]=414,AnniCard;
//
8.[212]=441,T-D1 XTRA;
8.[+T-D1 XTRA]=441,T-D1 XTRA;
8.[+T-D1]=441,T-D1 XTRA;
8.[+XTRA]=441,T-D1 XTRA;
//
8.[191]=479,Vodafone CallYa;
8.[+Vodafone CallYa]=479,Vodafone CallYa;
8.[+Vodafone]=479,Vodafone CallYa;
8.[+CallYa]=479,Vodafone CallYa;
//
8.[190]=480,E-Plus Free&Easy;
8.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
8.[+E-Plus]=480,E-Plus Free&Easy;
8.[+Free&Easy]=480,E-Plus Free&Easy;
//
8.[188]=481,O2 LOOP;
8.[+O2 LOOP]=481,O2 LOOP;
8.[+O2]=481,O2 LOOP;
8.[+LOOP]=481,O2 LOOP;
//
8.[+431]=431,Premiere Flex;
8.[+Premiere Flex]=431,Premiere Flex;
8.[+Premiere]=431,Premiere Flex;
8.[+Flex]=431,Premiere Flex;
//
8.[189]=405,gobananas;
8.[+gobananas]=405,gobananas;
//
8.[+352]=452,Paysafecard;
8.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
8.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
8.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
8.[807]=484,SCHLECKER PREPAID;
8.[249]=486,smobil;
8.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
8.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
8.[246]=488,FONIC;
8.[206]=489,BILDmobil;
8.[205]=490,Blau.de;
8.[207]=491,Congstar;
8.[475]=492,Lebara Mobile;
8.[248]=493,Simyo;
8.[884]=494,Lycamobile;
8.[917]=495,GT mobile;
// AVS -----------------------------
8.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
8.[+6010822]=497,Alnatura Gutscheinkarte;
// 220 TLAPA-KARTE---------------------
8.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
8.[+67]=1;
// COMFORT CARD -----------------------
8.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
8.[+5]=2;
8.[+222]=2;
8.[+223]=2;
8.[+224]=2;
8.[+225]=2;
8.[+226]=2;
8.[+227]=2;
8.[+228]=2;
8.[+229]=2;
8.[+230]=2;
8.[+231]=2;
8.[+232]=2;
8.[+233]=2;
8.[+234]=2;
8.[+235]=2;
8.[+236]=2;
8.[+237]=2;
8.[+238]=2;
8.[+239]=2;
8.[+240]=2;
8.[+241]=2;
8.[+242]=2;
8.[+243]=2;
8.[+244]=2;
8.[+245]=2;
8.[+246]=2;
8.[+247]=2;
8.[+248]=2;
8.[+249]=2;
8.[+250]=2;
8.[+251]=2;
8.[+252]=2;
8.[+253]=2;
8.[+254]=2;
8.[+255]=2;
8.[+256]=2;
8.[+257]=2;
8.[+258]=2;
8.[+259]=2;
8.[+261]=2;
8.[+262]=2;
8.[+263]=2;
8.[+264]=2;
8.[+265]=2;
8.[+266]=2;
8.[+267]=2;
8.[+268]=2;
8.[+269]=2;
8.[+270]=2;
8.[+271]=2;
8.[+272]=2;
// DINERS -----------------------------
8.[+36]=4;
// VISA -------------------------------
8.[+4]=5;
// AMEX -------------------------------
8.[+37]=6;
// JCB --------------------------------
8.[+35]=3;
// Alipay ---------------------
8.[97]=17,AliPay;
// Bluecode --------------------------------
8.[260]=22,BLUECODE;
//
// **********************************************************************************************************************************************
// Terminal type 9 (DE INGENICO 5300 ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
9.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
9.[5]=9,GIROCARD;
// EAPS -------------------------------
9.[7]=8,EAPS;
// Lastschrift ------
9.[9]=409,LASTSCHRIFT;
9.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
9.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
9.[13]=11,V PAY;
// BSW --------------------------------
9.[40]=404,BSW;
// MAESTRO ----------------------------
9.[21]=1;
// MASTERCARD -------------------------
9.[1]=2;
// DINERS -----------------------------
9.[4]=4;
// VISA -------------------------------
9.[3]=5;
// AMEX -------------------------------
9.[2]=6;
// MAESTRO DE / ELV -------------------
9.[0ELV]=410;
// MAESTRO DE / OLV -------------------
9.[0OLV]=411;
// MAESTRO DE / ec-CASH ---------------
9.[0CASH]=412;
// MAESTRO DE / default ---------------
9.[0]=401;
// GELDKARTE --------------------------
9.[30]=402;
// Comfort Card -----------------------
9.[42]=442,Comfort Card;
// Breuninger-- -----------------------
9.[38]=438,Breuninger;
// INTERCARD GUTSCHEINKARTE------------
9.[240]=440,IC-Gutscheinkarte;
//
// TEST PINPRINTING----------------
9.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
9.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
9.[807]=484,SCHLECKER PREPAID;
9.[249]=486,smobil;
9.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
9.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
9.[246]=488,FONIC;
9.[206]=489,BILDmobil;
9.[205]=490,Blau.de;
9.[207]=491,Congstar;
9.[475]=492,Lebara Mobile;
9.[248]=493,Simyo;
9.[884]=494,Lycamobile;
9.[917]=495,GT mobile;
// AVS -----------------------------
9.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
9.[+6010822]=497,Alnatura Gutscheinkarte;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
9.[+67]=1;
// COMFORT CARD -----------------------
9.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
9.[+5]=2;
9.[+222]=2;
9.[+223]=2;
9.[+224]=2;
9.[+225]=2;
9.[+226]=2;
9.[+227]=2;
9.[+228]=2;
9.[+229]=2;
9.[+230]=2;
9.[+231]=2;
9.[+232]=2;
9.[+233]=2;
9.[+234]=2;
9.[+235]=2;
9.[+236]=2;
9.[+237]=2;
9.[+238]=2;
9.[+239]=2;
9.[+240]=2;
9.[+241]=2;
9.[+242]=2;
9.[+243]=2;
9.[+244]=2;
9.[+245]=2;
9.[+246]=2;
9.[+247]=2;
9.[+248]=2;
9.[+249]=2;
9.[+250]=2;
9.[+251]=2;
9.[+252]=2;
9.[+253]=2;
9.[+254]=2;
9.[+255]=2;
9.[+256]=2;
9.[+257]=2;
9.[+258]=2;
9.[+259]=2;
9.[+261]=2;
9.[+262]=2;
9.[+263]=2;
9.[+264]=2;
9.[+265]=2;
9.[+266]=2;
9.[+267]=2;
9.[+268]=2;
9.[+269]=2;
9.[+270]=2;
9.[+271]=2;
9.[+272]=2;
// DINERS -----------------------------
9.[+36]=4;
// VISA -------------------------------
9.[+4]=5;
// AMEX -------------------------------
9.[+37]=6;
// JCB --------------------------------
9.[+35]=3;
// Alipay ---------------------
9.[97]=17,AliPay;
// Bluecode --------------------------------
9.[260]=22,BLUECODE;
//
// **********************************************************************************************************************************************
// Terminal type 10 (DE HYPERCOM ARTEMA DESK / ARTEMA HYBRID ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
10.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
10.[5]=9,GIROCARD;
// EAPS -------------------------------
10.[7]=8,EAPS;
// Lastschrift ------
10.[9]=409,LASTSCHRIFT;
10.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
10.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
10.[13]=11,V PAY;
// BSW --------------------------------
10.[40]=404,BSW;
// MAESTRO ----------------------------
10.[46]=1;
// MASTERCARD -------------------------
10.[6]=2;
// DINERS -----------------------------
10.[12]=4;
// VISA -------------------------------
10.[10]=5;
// AMEX -------------------------------
10.[8]=6;
// JCB --------------------------------
10.[14]=3;
// MAESTRO DE / ELV -------------------
10.[2ELV]=410;
// MAESTRO DE / OLV -------------------
10.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
10.[2CASH]=412;
// MAESTRO DE / default ---------------
10.[2]=401;
// GELDKARTE --------------------------
10.[30]=402;
// HORNBACH Profi ---------------------
10.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
10.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
10.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
10.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
10.[42]=442,Comfort Card;
// Breuninger -------------------------
10.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
10.[+414]=414,AnniCard;
10.[+AnniCard]=414,AnniCard;
//
10.[212]=441,T-D1 XTRA;
10.[80]=441,T-D1 XTRA;
10.[+T-D1 XTRA]=441,T-D1 XTRA;
10.[+T-D1]=441,T-D1 XTRA;
10.[+XTRA]=441,T-D1 XTRA;
//
10.[191]=479,Vodafone CallYa;
10.[+Vodafone CallYa]=479,Vodafone CallYa;
10.[+Vodafone]=479,Vodafone CallYa;
10.[+CallYa]=479,Vodafone CallYa;
//
10.[190]=480,E-Plus Free&Easy;
10.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
10.[+E-Plus]=480,E-Plus Free&Easy;
10.[+Free&Easy]=480,E-Plus Free&Easy;
//
10.[188]=481,O2 LOOP;
10.[+O2 LOOP]=481,O2 LOOP;
10.[+O2]=481,O2 LOOP;
10.[+LOOP]=481,O2 LOOP;
//
10.[+431]=431,Premiere Flex;
10.[+Premiere Flex]=431,Premiere Flex;
10.[+Premiere]=431,Premiere Flex;
10.[+Flex]=431,Premiere Flex;
//
10.[189]=405,gobananas;
10.[+gobananas]=405,gobananas;
//
10.[180]=452,Paysafecard Blue;
10.[181]=452,Paysafecard Red;
10.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
10.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
10.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
10.[807]=484,SCHLECKER PREPAID;
10.[249]=486,smobil;
10.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
10.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
10.[246]=488,FONIC;
10.[206]=489,BILDmobil;
10.[205]=490,Blau.de;
10.[207]=491,Congstar;
10.[475]=492,Lebara Mobile;
10.[248]=493,Simyo;
10.[884]=494,Lycamobile;
10.[917]=495,GT mobile;
// AVS -----------------------------
10.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
10.[+6010822]=497,Alnatura Gutscheinkarte;
//
// 220 TLAPA-KARTE---------------------
10.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
10.[+67]=1;
// COMFORT CARD -----------------------
10.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
10.[+5]=2;
10.[+222]=2;
10.[+223]=2;
10.[+224]=2;
10.[+225]=2;
10.[+226]=2;
10.[+227]=2;
10.[+228]=2;
10.[+229]=2;
10.[+230]=2;
10.[+231]=2;
10.[+232]=2;
10.[+233]=2;
10.[+234]=2;
10.[+235]=2;
10.[+236]=2;
10.[+237]=2;
10.[+238]=2;
10.[+239]=2;
10.[+240]=2;
10.[+241]=2;
10.[+242]=2;
10.[+243]=2;
10.[+244]=2;
10.[+245]=2;
10.[+246]=2;
10.[+247]=2;
10.[+248]=2;
10.[+249]=2;
10.[+250]=2;
10.[+251]=2;
10.[+252]=2;
10.[+253]=2;
10.[+254]=2;
10.[+255]=2;
10.[+256]=2;
10.[+257]=2;
10.[+258]=2;
10.[+259]=2;
10.[+261]=2;
10.[+262]=2;
10.[+263]=2;
10.[+264]=2;
10.[+265]=2;
10.[+266]=2;
10.[+267]=2;
10.[+268]=2;
10.[+269]=2;
10.[+270]=2;
10.[+271]=2;
10.[+272]=2;
// DINERS -----------------------------
10.[+36]=4;
// VISA -------------------------------
10.[+4]=5;
// AMEX -------------------------------
10.[+37]=6;
// JCB --------------------------------
10.[+35]=3;
// Alipay ---------------------
10.[97]=17,AliPay;
// Bluecode --------------------------------
10.[260]=22,BLUECODE;
//
// **********************************************************************************************************************************************
// Terminal type 11 (CH ACCARDA VIRTUAL CLIENT):
// The card recognition is hard coded.
// (The only possible card that can be used, is the ACCARDA Card)
// *******************************************************************************************************************************************
// ACCARDA ----------------------------
11.[+6004520306]=119,IKEA-CARD;
11.[+60045203]=119,Staedte-Card;
//
11.[+6004520698]=115,ACCARDA-Testcard;
11.[+60045206]=115,ACCARDA;
//
11.[115]=115,Giftcard;
// GROUPCARD (Loyalty Gift Card AG)------
11. [+a0000001574480]=186;
// **********************************************************************************************************************************************
// Terminal type 12 (BE ATOS ORIGIN):
// The card recognition is based on a BIT MAP OF APPLICATION ID'S or on the BRAND ID (VIC 1.07/11).
// *******************************************************************************************************************************************
// VISA Electron ----------------------
12.[114]=10;
12.[1002]=10;
12.[visa electron]=10;
// VISA -------------------------------
12.[124]=5;
12.[2002]=5;
12.[visa]=5;
// VISA VPay---------------------------
12.[82]=11;
12.[1003]=11;
12.[+vpay]=11;
12.[+v pay]=11;
// JCB --------------------------------
12.[120]=3;
12.[2007]=3;
12.[jcb]=3;
// DINERS -----------------------------
12.[121]=4;
12.[2005]=4;
12.[diners]=4;
// AMEX -------------------------------
12.[122]=6;
12.[2004]=6;
12.[american express]=6;
// MASTERCARD -------------------------
12.[123]=2;
12.[2003]=2;
12.[mastercard]=2;
// Maestro ----------------------------
12.[127]=1;
12.[1009]=1;
12.[maestro]=1;
// Bancontact/Mister Cash -------------
12.[128]=500;
12.[1001]=500;
12.[bancontact/mca]=500;
12.[bancontact]=500;
12.[tina]=500,TINA;
12.[100]=500,TINA;
// PROTON -----------------------------
12.[125]=501;
12.[1010]=501;
12.[proton]=501;
// AURORA -----------------------------
12.[109]=502;
12.[3001]=502;
12.[aurora]=502;
// KADOZ ------------------------------
12.[64]=503;
12.[kadoz]=503;
// CHINA UNION PAY --------------------
12.[2008]=12;
//
// **********************************************************************************************************************************************
// Terminal type 13 (TREIBAUF SIMPATICO):
// Do not set a card recognition. Card type should be set in Simpatico itself.
// **********************************************************************************************************************************************
// SIMPATICO Cards --------------------
//
// **********************************************************************************************************************************************
// Terminal type 14 (CH PAYSYS):
// The card recognition is based on ISSUER NUMBER, CARD NUMBER or APPLICATION ID.
// **********************************************************************************************************************************************
// CHINA UNION PAY --------------------
14.[9]=12,CHINA UNION PAY;
14.[09]=12,CHINA UNION PAY;
14.[a000000157446003]=12,CHINA UNION PAY;
14.[a000000157446004]=12,CHINA UNION PAY;
14.[a000000157446005]=12,CHINA UNION PAY;
14.[a000000157446006]=12,CHINA UNION PAY;
14.[a000000157446009]=12,CHINA UNION PAY;
14.[a0000001574460]=12,CHINA UNION PAY;
14.[a000000333010101]=12,CHINA UNION PAY;
14.[a000000333010102]=12,CHINA UNION PAY;
14.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO ----------------------------
14.[7]=1;
14.[07]=1;
// MASTERCARD -------------------------
14.[2]=2;
14.[02]=2;
// JCB --------------------------------
14.[6]=3;
14.[06]=3;
// DINERS -----------------------------
14.[43]=4;
// VISA -------------------------------
14.[5]=5;
14.[05]=5;
// V PAY -------------------------------
14.[96]=11,V PAY;
// AMEX -------------------------------
14.[40]=6;
// MAESTRO CH -------------------------
14.[1]=100;
14.[01]=100;
// M-CARD -----------------------------
14.[22]=101;
// POSTCARD ---------------------------
14.[21]=102,Postfinance Card;
// BERN CARD --------------------------
14.[41]=103;
14.[42]=103;
14.[45]=103;
// MOEVENPICK -------------------------
14.[46]=108;
// LOKALKARTE (READ OVER USB-PORT)------
14.[80]=135;
// JELMOLI SHOPPING BONUS CARD --------
14.[81]=110;
// MYONE ------------------------------
14.[84]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
14.[24]=116;
// POWERCARD --------------------------
14.[47]=117;
14.[a0000001574476]=117;
14.[a0000001574478]=117;
14.[a0000001574479]=117;
// COOP SUPERCARD PLUS VISA-------------
14.[26]=134;
// REKA  -------------------------------
14.[90]=120;
14.[+a000000157010E]=120;
14.[d7560000010101]=120;
14.[a0000001880101]=120;
14.[a0000001574464]=120;
14.[d7560000011111]=120;
// REKA RAIL --------------------------
14.[45]=180;
14.[d7560000011112]=180;
14.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
14.[78]=178;
14.[d7560000011113]=178;
14.[d7560000010102]=178;
14.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
14.[a000000157448C]=181;
// BONCARD -----------------------------
14.[66]=160,BONARD POINTS;
14.[67]=161,BONCARD PAY;
// LUNCH-CHECK -------------------------
14.[37]=172;
14.[a000000157447d]=172;
14.[d7560001150001]=172;
// CCV PAY -----------------------------
14.[35]=173;
// COOP EKZ ----------------------------
14.[37]=174;
14.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
14.[60]=175;
14.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
14.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
14.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
14.[a0000001574486]=179;
//if recognition by Issuer Number fails, try by card number prefix
// MASTERCARD -------------------------
14.[+50]=2;
14.[+51]=2;
14.[+52]=2;
14.[+53]=2
14.[+54]=2;
14.[+55]=2;
14.[+222]=2;
14.[+223]=2;
14.[+224]=2;
14.[+225]=2;
14.[+226]=2;
14.[+227]=2;
14.[+228]=2;
14.[+229]=2;
14.[+230]=2;
14.[+231]=2;
14.[+232]=2;
14.[+233]=2;
14.[+234]=2;
14.[+235]=2;
14.[+236]=2;
14.[+237]=2;
14.[+238]=2;
14.[+239]=2;
14.[+240]=2;
14.[+241]=2;
14.[+242]=2;
14.[+243]=2;
14.[+244]=2;
14.[+245]=2;
14.[+246]=2;
14.[+247]=2;
14.[+248]=2;
14.[+249]=2;
14.[+250]=2;
14.[+251]=2;
14.[+252]=2;
14.[+253]=2;
14.[+254]=2;
14.[+255]=2;
14.[+256]=2;
14.[+257]=2;
14.[+258]=2;
14.[+259]=2;
14.[+261]=2;
14.[+262]=2;
14.[+263]=2;
14.[+264]=2;
14.[+265]=2;
14.[+266]=2;
14.[+267]=2;
14.[+268]=2;
14.[+269]=2;
14.[+270]=2;
14.[+271]=2;
14.[+272]=2;
// MAESTRO ----------------------------
14.[+67]=1;
14.[+56]=1;
14.[+57]=1;
14.[+58]=1;
14.[+59]=1;
// VISA -------------------------------
14.[+4]=5;
// DINERS -----------------------------
14.[+36]=4;
14.[+38]=4;
14.[+39]=4;
// AMEX -------------------------------
14.[+34]=6;
14.[+37]=6;
// JCB --------------------------------
14.[+35]=3;
// GROUPCARD (Loyalty Gift Card AG)------
14.[+a0000001574480]=186;
// SWISS PAY
14.[a0000008010001]=2000,SWISS PAY;
14.[37]=2000,SWISS PAY;
// SWISS PAY 2
14.[a00000015744BD]=2001,SWISS PAY 2;
14.[37]=2001,SWISS PAY 2;
// **********************************************************************************************************************************************
// Terminal type 15 (CH ADUNO OPTIMUM):
// The card recognition is based on the ISSUER NUMBER and Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
15.[7]=1;
15.[07]=1;
15.[+a0000000043060]=1;
15.[+a0000001570021]=1;
// MASTERCARD -------------------------
15.[2]=2;
15.[02]=2;
15.[+a0000000041010]=2;
15.[+a0000001570020]=2;
// M-Budget
15.[a0000000000008]=2;
// JCB -------------------------------
15.[6]=3;
15.[06]=3;
15.[+a0000000651010]=3;
15.[+a0000001570040]=3;
// DINERS -----------------------------
15.[43]=4;
15.[+a0000001884443]=4;
15.[+a0000001574443]=4;
15.[+a0000001570060]=4;
15.[+a0000001523010]=4;
// VISA -------------------------------
15.[+a0000000031010]=5;
15.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
15.[+a0000000032010]=10;
15.[+a0000001570031]=10;
// V PAY -------------------------------
15.[96]=11,V PAY;
15.[a0000000032020]=11,V PAY;
15.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
15.[40]=6;
15.[+a00000002501]=6;
15.[+a0000001570010]=6;
15.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
15.[9]=12,CHINA UNION PAY;
15.[09]=12,CHINA UNION PAY;
15.[a000000157446003]=12,CHINA UNION PAY;
15.[a000000157446004]=12,CHINA UNION PAY;
15.[a000000157446005]=12,CHINA UNION PAY;
15.[a000000157446006]=12,CHINA UNION PAY;
15.[a000000157446009]=12,CHINA UNION PAY;
15.[a0000001574460]=12,CHINA UNION PAY;
15.[a000000333010101]=12,CHINA UNION PAY;
15.[a000000333010102]=12,CHINA UNION PAY;
15.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
15.[1]=100;
15.[01]=100;
15.[+a0000000101030]=100;
15.[+a0000001570022]=100;
15.[a0000000031030]=100;
// M-CARD -----------------------------
15.[22]=101;
15.[a0000001570100]=101;
15.[a0000001574450]=101;
15.[a000000162010201]=101;
15.[m-card]=101;
// POSTFINANCE CARD -------------------
15.[21]=102;
15.[a0000001110101]=102;
15.[a0000001999999]=102;
15.[a0000001999998]=102;
15.[a0000001570051]=102;
15.[a0000001570050]=102;
15.[a0000001574445]=102;
15.[a0000001574446]=102;
15.[a0000001570110]=143,PostFinance Mobile Coup.;
15.[a0000001570111]=144,PostFinance Mobile Paym.;
// BERN-CARD --------------------------
15.[42]=103;
15.[+a0000001570107]=103;
// ZUERI-CARD -------------------------
15.[41]=104;
15.[+a0000001570106]=104;
// BASLER-CARD ------------------------
15.[45]=105;
15.[+a0000001570108]=105;
// CASH -------------------------------
15.[+a0000001570023]=106;
// PHARMACARD --------------------------
15.[55]=107;
15.[+a0000001570105]=107;
// MOEVENPICK -------------------------
15.[46]=108;
15.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --------
15.[+a0000001570103]=110;
15.[+a000000157010B]=110;
15.[+a000000003000375]=110;
15.[81]=110;
// JELMOLI PAYCARD --------------------
15.[a0000001574462]=137;
// MANOR CARD (MYONE) -----------------
15.[84]=111;
15.[+a0000001570104]=111;
15.[+a0000001570109]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
15.[24]=116;
15.[a0000001574444]=116;
15.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
15.[26]=134;
15.[a0000001574458]=134;
15.[a00000000300037562]=134;
15.[a0000000037562]=134;
// POWERCARD (MFGroup)-----------------
15.[47]=117;
15.[+a000000157010D]=117;
15.[a0000001574476]=117;
15.[a0000001574478]=117;
15.[a0000001574479]=117;
// WIR --------------------------------
15.[85]=118;
15.[+a000000157010C]=118;
// REKA  -------------------------------
15.[90]=120;
15.[+a000000157010E]=120;
15.[d7560000010101]=120;
15.[a0000001880101]=120;
15.[a0000001574464]=120;
15.[d7560000011111]=120;
// REKA RAIL --------------------------
15.[45]=180;
15.[d7560000011112]=180;
15.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
15.[78]=178;
15.[d7560000011113]=178;
15.[d7560000010102]=178;
15.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
15.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
15.[+a000000157444E]=122,J-Geschenkkarte;
// ADUNO CARDS ------------------------
15.[+a0000001574448]=123,Aduno Kundenkarte;
15.[+a0000001574449]=124,Swiss Bonus Card;
15.[+a000000157444A]=125,ABO Card;
15.[a000000157010F]=142,ADUNO TEST;
// MEDIAMARKT -------------------------
15.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
15.[d7560000300101]=132,M-Budget;
// GIFT CARD (SIX)-----------------------
15.[a0000001574451]=133,Gift Card;
// SWISSCADEAU --------------------------
15.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
15.[a000000157445E]=140,BonGenie/Grieder Card; 
// GIFT CARD (BILLAG) -------------------
15.[a000000157010A]=141,GIFT CARD BILLAG;
// ACCARDA STAEDTEKARTE -----------------
15.[a0000001570112]=115,ACCARDA STAEDTEKARTE;
// KUNDENKARTE OFAC ---------------------
15.[a0000001570113]=145,Kundenkarte OFAC;
// REINVEST SELECT AG COMPANY CARD -----
15.[a0000001574447]=146,REINVEST SELECT;
// KUNDENKARTE PKZ -----------------------
15.[a000000157445D]=147,KUNDENKARTE PKZ; 
// JUBIN CARD ----------------------------
15.[a000000157445F]=148,JUBIN CARD;
// INNOCARD -------------------------------
15.[a0000001574448]=149,INNOCARD KUNDENKARTE;
15.[a000000157444A]=150,ABO CARD;
15.[a0000001574459]=151,SWISS HAPPY CARD;
15.[a0000001574461]=153,DATA STANDARDS;
15.[a0000001574463]=154,SWISSBONUS+;
// GE MONEY BANK ---------------------------
15.[a000000157444B]=156,GE MONEY BANK;
15.[a000000157444F]=157,BARRACUDA TEMPCARD;
// MERGER CARD -----------------------------
15.[a000000157444C]=158,MERGER CARD;
15.[a000000157444D]=159,MERGER CARD 2;
15.[a0000004840001]=158,MERGER CARD;
// BANCOMAT LUXEMBOURG -------------------
15.[a0000001574453]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ---------------------
15.[a0000001574454]=1001,AURORA LUXEMBOURG;
// BONCARD -------------------------------
15.[a0000001574455]=160,BONCARD POINTS;
15.[a000000157445B]=161,BONCARD PAY;
// FNAC ----------------------------------
15.[a0000001574456]=162,FNAC-KARTE;
// COMMIT CARD COMPANY CARD --------------
15.[a0000001574457]=163,COMMIT CARD;
// KUNDENKARTE LOEB ----------------------
15.[a000000157445C]=164,KUNDENKARTE LOEB;
// LUNCH-CHECK ---------------------------
15.[37]=172;
15.[a000000157447d]=172;
15.[d7560001150001]=172;
// CCV PAY -------------------------------
15.[35]=173;
15.[a000000157447b]=173;
// COOP EKZ ------------------------------
15.[37]=174;
15.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
15.[60]=175;
15.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
15.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
15.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
15.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
15.[+a0000001574480]=186;
// SWISS PAY
15.[a0000008010001]=2000,SWISS PAY;
15.[37]=2000,SWISS PAY;
// SWISS PAY 2
15.[a00000015744BD]=2001,SWISS PAY 2;
15.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 16 (US VERIFONE PCCHARGE):
// The card recognition is based on BIN Range (=PAN Prefix)
// **********************************************************************************************************************************************
// DCCB - Diners Club -----------------
16.[+300]=4,DINERS;
16.[+301]=4,DINERS;
16.[+302]=4,DINERS;
16.[+303]=4,DINERS;
16.[+304]=4,DINERS;
16.[+305]=4,DINERS;
16.[+3095]=4,DINERS;
16.[+3095]=4,DINERS;
16.[+3096]=4,DINERS;
16.[+3097]=4,DINERS;
16.[+3098]=4,DINERS;
16.[+3099]=4,DINERS;
16.[+36]=4,DINERS;
16.[+38]=4,DINERS;
16.[+39]=4,DINERS; 
16.[+60110]=4,DINERS;
16.[+60112]=4,DINERS;
16.[+60113]=4,DINERS;
16.[+60114]=4,DINERS;
16.[+601174]=4,DINERS;
16.[+601177]=4,DINERS;
16.[+601178]=4,DINERS;
16.[+601179]=4,DINERS;
16.[+601186]=4,DINERS;
16.[+601187]=4,DINERS;
16.[+601188]=4,DINERS;
16.[+601189]=4,DINERS;
16.[+60119]=4,DINERS;
16.[+644]=4,DINERS;
16.[+645]=4,DINERS;
16.[+646]=4,DINERS;
16.[+647]=4,DINERS;
16.[+648]=4,DINERS;
16.[+649]=4,DINERS;
16.[+65,-6541,-6556]=4,DINERS;  
// DISCOVER ---------------------------
16.[+6011]=600,DISCOVER;
16.[+6541]=600,DISCOVER;
16.[+6556]=600,DISCOVER; 
// MASTERCARD -------------------------
16.[+500235]=2,MASTERCARD;
16.[+500766]=2,MASTERCARD;
16.[+501012]=2,MASTERCARD;
16.[+502029]=2,MASTERCARD;
16.[+503615]=2,MASTERCARD;
16.[+504507]=2,MASTERCARD;
16.[+504834]=2,MASTERCARD;
16.[+504837]=2,MASTERCARD;
16.[+5049]=2,MASTERCARD;
16.[+51]=2,MASTERCARD;
16.[+52]=2,MASTERCARD;
16.[+53]=2,MASTERCARD;
16.[+54,-545250]=2,MASTERCARD;
16.[+55]=2,MASTERCARD; 
16.[+222]=2,MASTERCARD;
16.[+223]=2,MASTERCARD;
16.[+224]=2,MASTERCARD;
16.[+225]=2,MASTERCARD;
16.[+226]=2,MASTERCARD;
16.[+227]=2,MASTERCARD;
16.[+228]=2,MASTERCARD;
16.[+229]=2,MASTERCARD;
16.[+230]=2,MASTERCARD;
16.[+231]=2,MASTERCARD;
16.[+232]=2,MASTERCARD;
16.[+233]=2,MASTERCARD;
16.[+234]=2,MASTERCARD;
16.[+235]=2,MASTERCARD;
16.[+236]=2,MASTERCARD;
16.[+237]=2,MASTERCARD;
16.[+238]=2,MASTERCARD;
16.[+239]=2,MASTERCARD;
16.[+240]=2,MASTERCARD;
16.[+241]=2,MASTERCARD;
16.[+242]=2,MASTERCARD;
16.[+243]=2,MASTERCARD;
16.[+244]=2,MASTERCARD;
16.[+245]=2,MASTERCARD;
16.[+246]=2,MASTERCARD;
16.[+247]=2,MASTERCARD;
16.[+248]=2,MASTERCARD;
16.[+249]=2,MASTERCARD;
16.[+250]=2,MASTERCARD;
16.[+251]=2,MASTERCARD;
16.[+252]=2,MASTERCARD;
16.[+253]=2,MASTERCARD;
16.[+254]=2,MASTERCARD;
16.[+255]=2,MASTERCARD;
16.[+256]=2,MASTERCARD;
16.[+257]=2,MASTERCARD;
16.[+258]=2,MASTERCARD;
16.[+259]=2,MASTERCARD;
16.[+261]=2,MASTERCARD;
16.[+262]=2,MASTERCARD;
16.[+263]=2,MASTERCARD;
16.[+264]=2,MASTERCARD;
16.[+265]=2,MASTERCARD;
16.[+266]=2,MASTERCARD;
16.[+267]=2,MASTERCARD;
16.[+268]=2,MASTERCARD;
16.[+269]=2,MASTERCARD;
16.[+270]=2,MASTERCARD;
16.[+271]=2,MASTERCARD;
16.[+272]=2,MASTERCARD;
// CHINA UNION PAY --------------------
16.[+456351]=12,CHINA UNION PAY;
16.[+602969]=12,CHINA UNION PAY;
16.[+603367]=12,CHINA UNION PAY;
16.[+601382 ]=12,CHINA UNION PAY;
16.[+620]=12,CHINA UNION PAY;
16.[+621]=12,CHINA UNION PAY;
16.[+622]=12,CHINA UNION PAY;
16.[+623]=12,CHINA UNION PAY;
16.[+624]=12,CHINA UNION PAY;
16.[+625]=12,CHINA UNION PAY; 
// MAESTRO ----------------------------
16.[+50]=1,MAESTRO;
16.[+54250]=1,MAESTRO;
16.[+56]=1,MAESTRO;
16.[+57]=1,MAESTRO;
16.[+58]=1,MAESTRO;
16.[+59]=1,MAESTRO;
16.[+60]=1,MAESTRO;
16.[+61]=1,MAESTRO;
16.[+62]=1,MAESTRO;
16.[+63]=1,MAESTRO;
16.[+64]=1,MAESTRO;
16.[+66]=1,MAESTRO;
16.[+67]=1,MAESTRO;
16.[+68]=1,MAESTRO;
16.[+69]=1,MAESTRO;
// VISA ELECTRON -----------------------
16.[+400115]=10,VISA ELECTRON;
16.[+400837]=10,VISA ELECTRON;
16.[+400838]=10,VISA ELECTRON;
16.[+400839]=10,VISA ELECTRON;
16.[+412921]=10,VISA ELECTRON;
16.[+412922]=10,VISA ELECTRON;
16.[+412923]=10,VISA ELECTRON;
16.[+417935]=10,VISA ELECTRON;
16.[+419740]=10,VISA ELECTRON;
16.[+419741]=10,VISA ELECTRON;
16.[+419773]=10,VISA ELECTRON;
16.[+419774]=10,VISA ELECTRON;
16.[+419775]=10,VISA ELECTRON;
16.[+424519]=10,VISA ELECTRON;
16.[+424962]=10,VISA ELECTRON;
16.[+424963]=10,VISA ELECTRON;
16.[+437860]=10,VISA ELECTRON;
16.[+444000]=10,VISA ELECTRON;
16.[+459472]=10,VISA ELECTRON;
16.[+484406]=10,VISA ELECTRON;
16.[+484407]=10,VISA ELECTRON;
16.[+484408]=10,VISA ELECTRON;
16.[+484409]=10,VISA ELECTRON;
16.[+484410]=10,VISA ELECTRON;
16.[+484411]=10,VISA ELECTRON;
16.[+484413]=10,VISA ELECTRON;
16.[+484414]=10,VISA ELECTRON;
16.[+484418]=10,VISA ELECTRON;
16.[+484419]=10,VISA ELECTRON;
16.[+484420]=10,VISA ELECTRON;
16.[+484421]=10,VISA ELECTRON;
16.[+484422]=10,VISA ELECTRON;
16.[+484423]=10,VISA ELECTRON;
16.[+484424]=10,VISA ELECTRON;
16.[+484425]=10,VISA ELECTRON;
16.[+484426]=10,VISA ELECTRON;
16.[+484428]=10,VISA ELECTRON;
16.[+484429]=10,VISA ELECTRON;
16.[+48443]=10,VISA ELECTRON;
16.[+48444]=10,VISA ELECTRON;
16.[+484450]=10,VISA ELECTRON;
16.[+484451]=10,VISA ELECTRON;
16.[+484452]=10,VISA ELECTRON;
16.[+484453]=10,VISA ELECTRON;
16.[+484454]=10,VISA ELECTRON;
16.[+484455]=10,VISA ELECTRON;
16.[+49173]=10,VISA ELECTRON;
16.[+49174]=10,VISA ELECTRON;
16.[+49175]=10,VISA ELECTRON;
// VISA -------------------------------
16.[+4]=5,VISA;
// AMEX -------------------------------
16.[+34]=6,AMERICAN EXPRESS;
16.[+37]=6,AMERICAN EXPRESS;
// JCB --------------------------------
16.[+35]=3,JCB; 
// CBLN - Carte Blanche ---------------
16.[+389]=601;
// JAL- Japan Airlines ----------------
16.[+1800]=7;
16.[+2131]=7;
// ENRT - EN ROUTE --------------------
16.[+201400]=602,EN ROUTE;
//
// **********************************************************************************************************************************************
// Terminal type 17 (CH SIX PAYMENT SERVICES VEZPLUS):
// The card recognition is based on the ISSUER NUMBER and Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
17.[7]=1;
17.[07]=1;
// MASTERCARD -------------------------
17.[2]=2;
17.[02]=2;
17.[44]=2;
17.[+a0000000041010]=2;
// M-Budget
17.[25]=2;
// JCB --------------------------------
17.[6]=3;
17.[06]=3;
17.[49]=3;
// DINERS -----------------------------
17.[43]=4;
// VISA -------------------------------
17.[5]=5;
17.[05]=5;
17.[61]=5;
17.[62]=5;
17.[+a0000000031010]=5;
// VISA ELECTRON ----------------------
17.[+a0000000032010]=10;
// V PAY ------------------------------
17.[96]=11,V PAY;
// AMEX -------------------------------
17.[40]=6;
17.[a000000025010402]=6;
17.[a0000000291010]=6;
// CUP --------------------------------
17.[9]=12,CHINA UNION PAY;
17.[09]=12,CHINA UNION PAY;
17.[a000000333010101]=12,CHINA UNION PAY;
17.[a000000333010102]=12,CHINA UNION PAY;
17.[a000000333010103]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
17.[1]=100;
17.[01]=100;
17.[a0000000031030]=100;
// M-CARD -----------------------------
17.[22]=101;
// POSTCARD ---------------------------
17.[21]=102;
// CASH -------------------------------
17.[8]=106;
17.[08]=106;
// JELMOLI BONUS SHOPPING CARD --------
//17.[+600451]=110;
17.[81]=110;
// JELMOLI PAYCARD --------------------
17.[a0000001574462]=137;
// MY ONE -----------------------------
17.[84]=111; 
// ORANGE -----------------------------
17.[75]=112;
// Sunrise ----------------------------
17.[76]=113;
// SWISSCOM ---------------------------
17.[77]=114;
// COOP SUPERCARD PLUS MASTERCARD-------
17.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
17.[26]=134;
17.[a0000000037562]=134;
// POWERCARD --------------------------
17.[87]=117;
17.[47]=117;
17.[a0000001574476]=117;
17.[a0000001574478]=117;
17.[a0000001574479]=117;
// WIR --------------------------------
17.[50]=118; 
// IKEA-CARD --------------------------
17.[+6004520306]=119,IKEA-CARD   ;
17.[91]=119,IKEA-CARD   ;
17.[63]=119,IKEA-CARD   ;
// REKA  -------------------------------
17.[90]=120;
17.[+a000000157010E]=120;
17.[d7560000010101]=120;
17.[a0000001880101]=120;
17.[a0000001574464]=120;
17.[d7560000011111]=120;
// REKA RAIL --------------------------
17.[45]=180;
17.[d7560000011112]=180;
17.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
17.[78]=178;
17.[d7560000011113]=178;
17.[d7560000010102]=178;
17.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
17.[a000000157448C]=181;
// STAEDTECARD ------------------------
17.[41]=121;
17.[+600452]=121;
// JGIFT (JELMOLI)---------------------
17.[72]=122;
// YALLO ------------------------------
17.[73]=127,Yallo;
// COOP MOBILE ------------------------
17.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
17.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
17.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
17.[82]=131,M-Budget Mobile;
// GIFTCARD (SIX)-----------------------
17.[97]=133,GiftCard;
// SWISSCADEAU
17.[a000000157445A]=136,SwissCadeau;
// BANCOMAT LUXEMBOURG ----------------
17.[30]=1000,BANCOMAT LUXEMBOURG;
17.[a0000001574453]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ------------------
17.[31]=1001,AURORA LUXEMBOURG;
17.[a0000001574454]=1001,AURORA LUXEMBOURG;
// KUNDENKARTE BONGENIE/GRIEDER ---------

17.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
17.[66]=160,BONARD POINTS;
17.[67]=161,BONCARD PAY;
// LUNCH-CHECK --------------------------
17.[37]=172;
17.[a000000157447d]=172;
17.[d7560001150001]=172;
// CCV PAY ------------------------------
17.[35]=173;
17.[a000000157447b]=173;
// COOP EKZ -----------------------------
17.[37]=174;
17.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
17.[60]=175;
17.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
17.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
17.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
17.[a0000001574486]=179;
// KUNDENKARTE LOEB ----------------------------
17.[64]=164;
// JELMOLI BONUS SHOPPING CARD --------
17.[81]=110;
// JPAYC (JELMOLI)---------------------
17.[65]=182,JPAYC;
// GROUPCARD (Loyalty Gift Card AG)------
17.[+a0000001574480]=186;
// Paymit -------------------------------
17.[a000000157449a]=185,PAYMIT;
17.[a000000162000501]=185,PAYMIT;
17.[a000000162000502-ff]=185,PAYMIT;
17.[28]=185,PAYMIT;
// TWINT -------------------------------
17.[a000000157449e]=184,TWINT;
// BONCARD -------------------------------
17.[a0000001574455]=160,BONCARD POINTS;
17.[a000000157445B]=161,BONCARD PAY;
// SWISS PAY
17.[a0000008010001]=2000,SWISS PAY;
17.[37]=2000,SWISS PAY;
// SWISS PAY 2
17.[a00000015744BD]=2001,SWISS PAY 2;
17.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 18 (DE HYPERCOM/THALES EFT Library (MCSSoft)):
// The card recognition is based on the ISSUER NUMBER and PAN
// **********************************************************************************************************************************************
// MAESTRO_DE -------------------------
18.[0]=401;
18.[+672]=401;
// MAESTRO ----------------------------
18.[21]=1;
18.[+56]=1;
18.[+57]=1;
18.[+58]=1;
18.[+59]=1;
18.[+6]=1;
// MASTERCARD -------------------------
18.[+222]=2;
18.[+223]=2;
18.[+224]=2;
18.[+225]=2;
18.[+226]=2;
18.[+227]=2;
18.[+228]=2;
18.[+229]=2;
18.[+230]=2;
18.[+231]=2;
18.[+232]=2;
18.[+233]=2;
18.[+234]=2;
18.[+235]=2;
18.[+236]=2;
18.[+237]=2;
18.[+238]=2;
18.[+239]=2;
18.[+240]=2;
18.[+241]=2;
18.[+242]=2;
18.[+243]=2;
18.[+244]=2;
18.[+245]=2;
18.[+246]=2;
18.[+247]=2;
18.[+248]=2;
18.[+249]=2;
18.[+250]=2;
18.[+251]=2;
18.[+252]=2;
18.[+253]=2;
18.[+254]=2;
18.[+255]=2;
18.[+256]=2;
18.[+257]=2;
18.[+258]=2;
18.[+259]=2;
18.[+261]=2;
18.[+262]=2;
18.[+263]=2;
18.[+264]=2;
18.[+265]=2;
18.[+266]=2;
18.[+267]=2;
18.[+268]=2;
18.[+269]=2;
18.[+270]=2;
18.[+271]=2;
18.[+272]=2;
// other cards have the Pepper default values and are handled internally
//
// **********************************************************************************************************************************************
// Terminal type 19 (SI SIX PAYMENT SERVICES MPD): 
// The card recognition is based on AID, ISSUER NUMBER, Card Number and Card Name
// **********************************************************************************************************************************************
// where Issuer-Nbr exist do not use AID, rely on Issuer-Nbr from terminal
// MAESTRO ----------------------------
19.[7]=1;
19.[07]=1;
// MASTERCARD -------------------------
19.[2]=2;
19.[02]=2;
19.[+a0000000041010]=2;
// JCB --------------------------------
19.[6]=3;
19.[06]=3;
// DINERS -----------------------------
19.[43]=4;
// VISA -------------------------------
19.[5]=5;
19.[05]=5;
19.[+a0000000031010]=5;
19.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
19.[+a0000000032010]=10,VISA ELECTRON;
19.[+a0000001570031]=10,VISA ELECTRON;
// V PAY -------------------------------
19.[96]=11,V PAY;
19.[a0000000032020]=11,V PAY;
19.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
19.[40]=6;
19.[a000000025010402]=6;
19.[a0000000291010]=6;
// BANCOMAT LUXEMBOURG ----------------
19.[30]=1000,BANCOMAT LUXEMBOURG;
19.[+D44200002542414E]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ------------------
19.[31]=1001,AURORA LUXEMBOURG;
// MAESTRO CH -------------------------
19.[1]=100;
19.[01]=100;
// CUP --------------------------------
19.[9]=12,CHINA UNION PAY;
19.[09]=12,CHINA UNION PAY;
19.[a000000333010101]=12,CHINA UNION PAY;
19.[a000000333010102]=12,CHINA UNION PAY;
// BONCARD ----------------------------
19.[66]=160,BONARD POINTS;
19.[67]=161,BONCARD PAY;
// LUNCH-CHECK -------------------------
19.[37]=172;
19.[a000000157447d]=172;
19.[d7560001150001]=172;
// CCV PAY -----------------------------
19.[35]=173;
19.[a000000157447b]=173;
// CO2NEUTRAL PAY -----------------------
19.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
19.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
19.[a0000001574486]=179;
//
// **********************************************************************************************************************************************
// Terminal type 20 (CH INGENICO CRP 2.42): 
// The card recognition is based on the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
20.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
20.[+a0000001570021]=1;
// MASTERCARD ---------------------
20.[+a0000000041010]=2;
20.[+a0000001570020]=2;
// M-Budget -----------------------
20.[a0000000000008]=2;
// JCB ---------------------------
20.[+a0000000651010]=3;
20.[+a0000001570040]=3;
// DINERS ------------------------
20.[+a0000001884443]=4;
20.[+a0000001574443]=4;
20.[+a0000001570060]=4;
20.[+a0000001523010]=4;
// VISA --------------------------
20.[+a0000000031010]=5;
20.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
20.[+a0000000032010]=10;
20.[+a0000001570031]=10;
// V PAY -------------------------------
20.[+a0000000032020]=11,V PAY;
20.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
20.[+a0000001570010]=6;
20.[+a00000002501]=6;
20.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
20.[a000000157446003]=12,CHINA UNION PAY;
20.[a000000157446004]=12,CHINA UNION PAY;
20.[a000000157446005]=12,CHINA UNION PAY;
20.[a000000157446006]=12,CHINA UNION PAY;
20.[a000000157446009]=12,CHINA UNION PAY;
20.[a0000001574460]=12,CHINA UNION PAY;
20.[a000000333010101]=12,CHINA UNION PAY;
20.[a000000333010102]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
20.[+a0000000101030]=100;
20.[+a0000001570022]=100;
20.[a0000000031030]=100;
// M-CARD ------------------------
20.[a0000001570100]=101;
20.[a0000001574450]=101;
// POSTCARD
20.[+a0000001110101]=102;
20.[+a0000001999999]=102;
20.[+a0000001999998]=102;
20.[+a0000001570051]=102;
20.[+a0000001570050]=102;
20.[+a0000001574445]=102;
20.[+a0000001574446]=102;
// BERN CARD --------------------
20.[+a0000001570107]=103;
// ZUERI CARD -------------------
20.[+a0000001570106]=104;
// BASLER CARD ------------------
20.[+a0000001570108]=105;
// CASH -------------------------
20.[+a0000001570023]=106;
// PHARMACARD --------------------
20.[+a0000001570105]=107;
// MOEVENPICK -------------------
20.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
20.[+a0000001570103]=110;
20.[+a000000157010B]=110;
20.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
20.[a0000001574462]=137;
// MY ONE -----------------------
20.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
20.[a0000001574444]=116;
20.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
20.[a0000001574458]=134;
20.[a00000000300037562]=134;
20.[a0000000037562]=134;
// POWER CARD -------------------
20.[+a000000157010D]=117;
20.[a0000001574476]=117;
20.[a0000001574478]=117;
20.[a0000001574479]=117;
// WIR --------------------------
20.[+a000000157010C]=118;
// REKA  ------------------------------
20.[+a000000157010E]=120;
20.[+D7560000010101]=120;
20.[d7560000011111]=120;
// REKA RAIL --------------------------
20.[d7560000011112]=180;
20.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
20.[d7560000011113]=178;
20.[d7560000010102]=178;
20.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
20.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
20.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
20.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
20.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
20.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
20.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
20.[a0000001574455]=160,BONARD POINTS;
20.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK --------------------------
20.[a000000157447d]=172;
20.[d7560001150001]=172;
// CCV PAY ------------------------------
20.[a000000157447b]=173;
// COOP EKZ -----------------------------
20.[37]=174;
20.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
20.[60]=175;
20.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
20.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
20.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
20.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
20.[+a0000001574480]=186;
// SWISS PAY
20.[a0000008010001]=2000,SWISS PAY;
20.[37]=2000,SWISS PAY;
// SWISS PAY 2
20.[a00000015744BD]=2001,SWISS PAY 2;
20.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 21 (INTERNATIONAL USE / SIX PAYMENT SERVICES MPD): 
// The card recognition is based on AID, ISSUER NUMBER, Card Number and Card Name
// **********************************************************************************************************************************************
// where Issuer-Nbr exist do not use AID, rely on Issuer-Nbr from terminal
// MAESTRO ----------------------------
21.[7]=1;
21.[07]=1;
// MASTERCARD -------------------------
21.[2]=2;
21.[02]=2;
21.[+a0000000041010]=2;
// JCB --------------------------------
21.[6]=3;
21.[06]=3;
// DINERS -----------------------------
21.[43]=4;
// VISA -------------------------------
//21.[5]=5;  // DO NOT USE BRAND-ID 5 because of mix up with Bancomat Luxembourg
//21.[05]=5;
21.[+a0000000031010]=5;
21.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
21.[+a0000000032010]=10,VISA ELECTRON;
21.[+a0000001570031]=10,VISA ELECTRON;
// V PAY -------------------------------
21.[96]=11,V PAY;
21.[a0000000032020]=11,V PAY;
21.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
21.[40]=6;
21.[a000000025010402]=6;
21.[a0000000291010]=6;
// BANCOMAT LUXEMBOURG ----------------
21.[30]=1000,BANCOMAT LUXEMBOURG;
21.[+D44200002542414E]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ------------------
21.[31]=1001,AURORA LUXEMBOURG;
// MAESTRO CH -------------------------
21.[1]=100;
21.[01]=100;
// CUP --------------------------------
21.[9]=12,CHINA UNION PAY;
21.[09]=12,CHINA UNION PAY;
21.[a000000333010101]=12,CHINA UNION PAY;
21.[a000000333010102]=12,CHINA UNION PAY;
// BONCARD ----------------------------
21.[66]=160,BONARD POINTS;
21.[67]=161,BONCARD PAY;
// LUNCH-CHECK ------------------------
21.[37]=172;
21.[a000000157447d]=172;
21.[d7560001150001]=172;
// CCV PAY ----------------------------
21.[35]=173;
21.[a000000157447b]=173;
// CO2NEUTRAL PAY -----------------------
21.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
21.[a0000001574482]=177;
// BO1CARD TEST (DIREKT) ----------------
21.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
21. [+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 22 (SI B+S Card Services CCV ST-4000 ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
22.[198]=12,CHINA UNION PAY;
//-----------------  new card types TA 7.0 --------------
// girocard ---------------------------
22.[5]=9,GIROCARD;
// EAPS -------------------------------
22.[7]=8,EAPS;
// girocard deutsche Lastschrift ------
22.[9]=409,GIROCARD DE LASTSCHRIFT;
22.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
22.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
22.[13]=11,V PAY;
//-------------- end new card types TA 7.0 -------------
// MAESTRO ----------------------------
22.[46]=1;
// MASTERCARD -------------------------
22.[6]=2;
// DINERS -----------------------------
22.[12]=4;
// VISA -------------------------------
22.[10]=5;
// AMEX -------------------------------
22.[8]=6;
// JCB --------------------------------
22.[14]=3;
// INTERCARD GUTSCHEINKARTE------------
22.[240]=440,IC-Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
22.[+6010822]=497,Alnatura Gutscheinkarte;
// 
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
22.[+67]=1;
// MASTERCARD -------------------------
22.[+5]=2;
22.[+222]=2;
22.[+223]=2;
22.[+224]=2;
22.[+225]=2;
22.[+226]=2;
22.[+227]=2;
22.[+228]=2;
22.[+229]=2;
22.[+230]=2;
22.[+231]=2;
22.[+232]=2;
22.[+233]=2;
22.[+234]=2;
22.[+235]=2;
22.[+236]=2;
22.[+237]=2;
22.[+238]=2;
22.[+239]=2;
22.[+240]=2;
22.[+241]=2;
22.[+242]=2;
22.[+243]=2;
22.[+244]=2;
22.[+245]=2;
22.[+246]=2;
22.[+247]=2;
22.[+248]=2;
22.[+249]=2;
22.[+250]=2;
22.[+251]=2;
22.[+252]=2;
22.[+253]=2;
22.[+254]=2;
22.[+255]=2;
22.[+256]=2;
22.[+257]=2;
22.[+258]=2;
22.[+259]=2;
22.[+261]=2;
22.[+262]=2;
22.[+263]=2;
22.[+264]=2;
22.[+265]=2;
22.[+266]=2;
22.[+267]=2;
22.[+268]=2;
22.[+269]=2;
22.[+270]=2;
22.[+271]=2;
22.[+272]=2;
// DINERS -----------------------------
22.[+36]=4;
// VISA -------------------------------
22.[+4]=5;
// AMEX -------------------------------
22.[+37]=6;
// JCB --------------------------------
22.[+35]=3;
// **********************************************************************************************************************************************
// Terminal type 23 (CZ INGENICO, CESKA SPORITELNA, POS-ECR interface Version 2.3):
// The card recognition is based on CARD ISSUER ID (CARD_TYPE).
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
23.[05]=1,CS and others;
// MASTERCARD -------------------------
23.[01]=2,MasterCard ;
// DINERS -----------------------------
23.[02]=4,Diners;
// VISA -------------------------------
23.[00]=5,VISA;
// AMEX -------------------------------
23.[04]=6,AmEx;
// JCB --------------------------------
23.[03]=3,JCB;
//
// **********************************************************************************************************************************************
// Terminal type 24 (FR CONCERT CAISSE EURO V. 2.0):
// The card recognition is based on the EMETTEUER and the CARD NUMBER PREFIX.
// **********************************************************************************************************************************************
// DINERS -------------------------------
24.[6]=4,DINERS CLUB FRANCE;
24.[+300]=4,DINERS;
24.[+301]=4,DINERS;
24.[+302]=4,DINERS;
24.[+303]=4,DINERS;
24.[+304]=4,DINERS;
24.[+305]=4,DINERS;
24.[+3095]=4,DINERS;
24.[+3095]=4,DINERS;
24.[+3096]=4,DINERS;
24.[+3097]=4,DINERS;
24.[+3098]=4,DINERS;
24.[+3099]=4,DINERS;
24.[+36]=4,DINERS;
24.[+38]=4,DINERS;
24.[+39]=4,DINERS;
24.[+60110]=4,DINERS;
24.[+60112]=4,DINERS;
24.[+60113]=4,DINERS;
24.[+60114]=4,DINERS;
24.[+601174]=4,DINERS;
24.[+601177]=4,DINERS;
24.[+601178]=4,DINERS;
24.[+601179]=4,DINERS;
24.[+601186]=4,DINERS;
24.[+601187]=4,DINERS;
24.[+601188]=4,DINERS;
24.[+601189]=4,DINERS;
24.[+60119]=4,DINERS;
24.[+644]=4,DINERS;
24.[+645]=4,DINERS;
24.[+646]=4,DINERS;
24.[+647]=4,DINERS;
24.[+648]=4,DINERS;
24.[+649]=4,DINERS;
24.[+65]=4,DINERS; 
// MASTERCARD ---------------------------
24.[+500235]=2,MASTERCARD;
24.[+500766]=2,MASTERCARD;
24.[+501012]=2,MASTERCARD;
24.[+502029]=2,MASTERCARD;
24.[+503615]=2,MASTERCARD;
24.[+504507]=2,MASTERCARD;
24.[+504834]=2,MASTERCARD;
24.[+504837]=2,MASTERCARD;
24.[+5049]=2,MASTERCARD;
24.[+51]=2,MASTERCARD;
24.[+52]=2,MASTERCARD;
24.[+53]=2,MASTERCARD;
24.[+54,-545250]=2,MASTERCARD;
24.[+55]=2,MASTERCARD;
24.[+222]=2,MASTERCARD;
24.[+223]=2,MASTERCARD;
24.[+224]=2,MASTERCARD;
24.[+225]=2,MASTERCARD;
24.[+226]=2,MASTERCARD;
24.[+227]=2,MASTERCARD;
24.[+228]=2,MASTERCARD;
24.[+229]=2,MASTERCARD;
24.[+230]=2,MASTERCARD;
24.[+231]=2,MASTERCARD;
24.[+232]=2,MASTERCARD;
24.[+233]=2,MASTERCARD;
24.[+234]=2,MASTERCARD;
24.[+235]=2,MASTERCARD;
24.[+236]=2,MASTERCARD;
24.[+237]=2,MASTERCARD;
24.[+238]=2,MASTERCARD;
24.[+239]=2,MASTERCARD;
24.[+240]=2,MASTERCARD;
24.[+241]=2,MASTERCARD;
24.[+242]=2,MASTERCARD;
24.[+243]=2,MASTERCARD;
24.[+244]=2,MASTERCARD;
24.[+245]=2,MASTERCARD;
24.[+246]=2,MASTERCARD;
24.[+247]=2,MASTERCARD;
24.[+248]=2,MASTERCARD;
24.[+249]=2,MASTERCARD;
24.[+250]=2,MASTERCARD;
24.[+251]=2,MASTERCARD;
24.[+252]=2,MASTERCARD;
24.[+253]=2,MASTERCARD;
24.[+254]=2,MASTERCARD;
24.[+255]=2,MASTERCARD;
24.[+256]=2,MASTERCARD;
24.[+257]=2,MASTERCARD;
24.[+258]=2,MASTERCARD;
24.[+259]=2,MASTERCARD;
24.[+261]=2,MASTERCARD;
24.[+262]=2,MASTERCARD;
24.[+263]=2,MASTERCARD;
24.[+264]=2,MASTERCARD;
24.[+265]=2,MASTERCARD;
24.[+266]=2,MASTERCARD;
24.[+267]=2,MASTERCARD;
24.[+268]=2,MASTERCARD;
24.[+269]=2,MASTERCARD;
24.[+270]=2,MASTERCARD;
24.[+271]=2,MASTERCARD;
24.[+272]=2,MASTERCARD;
// CHINA UNION PAY NATIXIS / CHINA UNION PAY
24.[+456351]=12,CHINA UNION PAY;
24.[+602969]=12,CHINA UNION PAY;
24.[+603367]=12,CHINA UNION PAY;
24.[+620]=12,CHINA UNION PAY;
24.[+621]=12,CHINA UNION PAY;
24.[+622]=12,CHINA UNION PAY;
24.[+623]=12,CHINA UNION PAY;
24.[+624]=12,CHINA UNION PAY;
24.[+625]=12,CHINA UNION PAY;
// MAESTRO ------------------------------
24.[+50]=1,MAESTRO;
24.[+54250]=1,MAESTRO;
24.[+56]=1,MAESTRO;
24.[+57]=1,MAESTRO;
24.[+58]=1,MAESTRO;
24.[+59]=1,MAESTRO;
24.[+60]=1,MAESTRO;
24.[+61]=1,MAESTRO;
24.[+62]=1,MAESTRO;
24.[+63]=1,MAESTRO;
24.[+64]=1,MAESTRO;
24.[+66]=1,MAESTRO;
24.[+67]=1,MAESTRO;
24.[+68]=1,MAESTRO;
24.[+69]=1,MAESTRO; 
// VISA ELECTRON -----------------------
24.[+400115]=10,VISA ELECTRON;
24.[+400837]=10,VISA ELECTRON;
24.[+400838]=10,VISA ELECTRON;
24.[+400839]=10,VISA ELECTRON;
24.[+412921]=10,VISA ELECTRON;
24.[+412922]=10,VISA ELECTRON;
24.[+412923]=10,VISA ELECTRON;
24.[+417935]=10,VISA ELECTRON;
24.[+419740]=10,VISA ELECTRON;
24.[+419741]=10,VISA ELECTRON;
24.[+419773]=10,VISA ELECTRON;
24.[+419774]=10,VISA ELECTRON;
24.[+419775]=10,VISA ELECTRON;
24.[+424519]=10,VISA ELECTRON;
24.[+424962]=10,VISA ELECTRON;
24.[+424963]=10,VISA ELECTRON;
24.[+437860]=10,VISA ELECTRON;
24.[+444000]=10,VISA ELECTRON;
24.[+459472]=10,VISA ELECTRON;
24.[+484406]=10,VISA ELECTRON;
24.[+484407]=10,VISA ELECTRON;
24.[+484408]=10,VISA ELECTRON;
24.[+484409]=10,VISA ELECTRON;
24.[+484410]=10,VISA ELECTRON;
24.[+484411]=10,VISA ELECTRON;
24.[+484413]=10,VISA ELECTRON;
24.[+484414]=10,VISA ELECTRON;
24.[+484418]=10,VISA ELECTRON;
24.[+484419]=10,VISA ELECTRON;
24.[+484420]=10,VISA ELECTRON;
24.[+484421]=10,VISA ELECTRON;
24.[+484422]=10,VISA ELECTRON;
24.[+484423]=10,VISA ELECTRON;
24.[+484424]=10,VISA ELECTRON;
24.[+484425]=10,VISA ELECTRON;
24.[+484426]=10,VISA ELECTRON;
24.[+484428]=10,VISA ELECTRON;
24.[+484429]=10,VISA ELECTRON;
24.[+48443]=10,VISA ELECTRON;
24.[+48444]=10,VISA ELECTRON;
24.[+484450]=10,VISA ELECTRON;
24.[+484451]=10,VISA ELECTRON;
24.[+484452]=10,VISA ELECTRON;
24.[+484453]=10,VISA ELECTRON;
24.[+484454]=10,VISA ELECTRON;
24.[+484455]=10,VISA ELECTRON;
24.[+49173]=10,VISA ELECTRON;
24.[+49174]=10,VISA ELECTRON;
24.[+49175]=10,VISA ELECTRON; 
// VISA ---------------------------------
24.[+4]=5,VISA;
// AMEX ---------------------------------
24.[2]=6,AMERICAN EXPRESS;
24.[+34]=6,AMERICAN EXPRESS;
24.[+37]=6,AMERICAN EXPRESS;
// JCB ----------------------------------
24.[9]=3,JCB;
24.[+35]=3,JCB;
// REFLEXE/AURORE - PROT. CETNCB CETDIS----
24.[3]=911,AURORE;
// CETELEM -------------------------------
24.[4]=931,CETELM;
// COFINOGA ------------------------------
24.[5]=908,COFINOGA; 
// PASS PISTE / PASS EMV------------------
24.[7]=923,PASS;
// PLURIEL - FRANFINANCE -----------------
24.[8]=915,FRANFINANCE;
// ACCORD --------------------------------
24.[A]=901,ACCORD;
// CHEQUE --------------------------------
24.[C]=909,CHEQUE;
// Finaref -------------------------------
24.[F]=917,FINAREF; 
// Modeus --------------------------------
24.[M]=932,MODEUS;
// CARTE PINAULT PRINTEMPS REDOUTE -------
24.[P]=933,PINAULT PRINTEMPS RED.;
//
// **********************************************************************************************************************************************
// Terminal type 25 (DE INGENICO U32 ZVT)
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
25.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
25.[5]=9,GIROCARD;
// EAPS -------------------------------
25.[7]=8,EAPS;
// Lastschrift ------
25.[9]=409,LASTSCHRIFT;
25.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
25.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
25.[13]=11,V PAY;
// BSW --------------------------------
25.[40]=404,BSW;
// AUSTRIA CARD -----------------------
25.[217]=217,Austria Card;
// MAESTRO ----------------------------
25.[46]=1;
// MASTERCARD -------------------------
25.[6]=2;
// DINERS -----------------------------
25.[12]=4;
// VISA -------------------------------
25.[10]=5;
// AMEX -------------------------------
25.[8]=6;
// JCB --------------------------------
25.[14]=3;
// MAESTRO DE / ELV -------------------
25.[2ELV]=410;
// MAESTRO DE / OLV -------------------
25.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
25.[2CASH]=412;
// MAESTRO DE / default ---------------
25.[2]=401;
// GELDKARTE --------------------------
25.[30]=402;
// IKEA FAMILY PLUS -------------------
25.[76]=403;
// HORNBACH Profi ---------------------
25.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
25.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
25.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
25.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
25.[42]=442,Comfort Card;
// Breuninger--------------------------
25.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
25.[+414]=414,AnniCard;
25.[+AnniCard]=414,AnniCard;
//
25.[212]=441,T-D1 XTRA;
25.[+T-D1 XTRA]=441,T-D1 XTRA;
25.[+T-D1]=441,T-D1 XTRA;
25.[+XTRA]=441,T-D1 XTRA;
//
25.[191]=479,Vodafone CallYa;
25.[+Vodafone CallYa]=479,Vodafone CallYa;
25.[+Vodafone]=479,Vodafone CallYa;
25.[+CallYa]=479,Vodafone CallYa;
//
25.[+80]=480,E-Plus Free&Easy;
25.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
25.[+E-Plus]=480,E-Plus Free&Easy;
25.[+Free&Easy]=480,E-Plus Free&Easy;
//
25.[188]=481,O2 LOOP;
25.[+O2 LOOP]=481,O2 LOOP;
25.[+O2]=481,O2 LOOP;
25.[+LOOP]=481,O2 LOOP;
//
25.[+431]=431,Premiere Flex;
25.[+Premiere Flex]=431,Premiere Flex;
25.[+Premiere]=431,Premiere Flex;
25.[+Flex]=431,Premiere Flex;
//
25.[189]=405,gobananas;
25.[+gobananas]=405,gobananas;
//
25.[+352]=452,Paysafecard;
25.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
25.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
25.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
25.[807]=484,SCHLECKER PREPAID;
25.[249]=486,smobil;
25.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
25.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
25.[246]=488,FONIC;
25.[206]=489,BILDmobil;
25.[205]=490,Blau.de;
25.[207]=491,Congstar;
25.[475]=492,Lebara Mobile;
25.[248]=493,Simyo;
25.[884]=494,Lycamobile;
25.[917]=495,GT mobile;
// AVS -----------------------------
25.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
25.[+6010822]=497,Alnatura Gutscheinkarte;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
25.[+67]=1;
// COMFORT CARD -----------------------
25.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
25.[+5]=2;
25.[+222]=2;
25.[+223]=2;
25.[+224]=2;
25.[+225]=2;
25.[+226]=2;
25.[+227]=2;
25.[+228]=2;
25.[+229]=2;
25.[+230]=2;
25.[+231]=2;
25.[+232]=2;
25.[+233]=2;
25.[+234]=2;
25.[+235]=2;
25.[+236]=2;
25.[+237]=2;
25.[+238]=2;
25.[+239]=2;
25.[+240]=2;
25.[+241]=2;
25.[+242]=2;
25.[+243]=2;
25.[+244]=2;
25.[+245]=2;
25.[+246]=2;
25.[+247]=2;
25.[+248]=2;
25.[+249]=2;
25.[+250]=2;
25.[+251]=2;
25.[+252]=2;
25.[+253]=2;
25.[+254]=2;
25.[+255]=2;
25.[+256]=2;
25.[+257]=2;
25.[+258]=2;
25.[+259]=2;
25.[+261]=2;
25.[+262]=2;
25.[+263]=2;
25.[+264]=2;
25.[+265]=2;
25.[+266]=2;
25.[+267]=2;
25.[+268]=2;
25.[+269]=2;
25.[+270]=2;
25.[+271]=2;
25.[+272]=2;
// DINERS -----------------------------
25.[+36]=4;
// VISA -------------------------------
25.[+4]=5;
// AMEX -------------------------------
25.[+37]=6;
// JCB --------------------------------
25.[+35]=3;
// Alipay ---------------------
25.[97]=17,AliPay;
// Bluecode --------------------------------
25.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 25 (DE INGENICO U32) >>>>>> Atos Card-IDs <<<<<<
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
//25.[21]=1;
// MASTERCARD -------------------------
//25.[1]=2;
// DINERS -----------------------------
//25.[4]=4;
// VISA -------------------------------
//25.[3]=5;
// AMEX -------------------------------
//25.[2]=6;
// MAESTRO DE / ELV -------------------
//25.[0ELV]=410;
// MAESTRO DE / OLV -------------------
//25.[0OLV]=411;
// MAESTRO DE / ec-CASH ---------------
//25.[0CASH]=412;
// MAESTRO DE / default ---------------
//25.[0]=401;
// GELDKARTE --------------------------
//25.[30]=402;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
//25.[+67]=1;
// COMFORT CARD -----------------------
//25.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
//25.[+5]=2;
// DINERS -----------------------------
//25.[+36]=4;
// VISA -------------------------------
//25.[+4]=5;
// AMEX -------------------------------
//25.[+37]=6;
// JCB --------------------------------
//25.[+35]=3;
// **********************************************************************************************************************************************
// Terminal type 26 (CH ADUNO VERDI):
// The card recognition is based on the ISSUER NUMBER or Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
26.[7]=1;
26.[07]=1;
26.[+a0000000043060]=1;
26.[+a0000001570021]=1;
// MASTERCARD -------------------------
26.[2]=2;
26.[02]=2;
26.[+a0000000041010]=2;
26.[+a0000001570020]=2;
// M-Budget
26.[a0000000000008]=2;
// JCB -------------------------------
26.[6]=3;
26.[06]=3;
26.[+a0000000651010]=3;
26.[+a0000001570040]=3;
// DINERS -----------------------------
26.[43]=4;
26.[+a0000001884443]=4;
26.[+a0000001574443]=4;
26.[+a0000001570060]=4;
26.[+a0000001523010]=4;
// VISA -------------------------------
26.[+a0000000031010]=5;
26.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
26.[+a0000000032010]=10;
26.[+a0000001570031]=10;
// V PAY -------------------------------
26.[96]=11,V PAY;
26.[a0000000032020]=11,V PAY;
26.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
26.[40]=6;
26.[+a00000002501]=6;
26.[+a0000001570010]=6;
26.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
26.[9]=12,CHINA UNION PAY;
26.[09]=12,CHINA UNION PAY;
26.[a000000157446003]=12,CHINA UNION PAY;
26.[a000000157446004]=12,CHINA UNION PAY;
26.[a000000157446005]=12,CHINA UNION PAY;
26.[a000000157446006]=12,CHINA UNION PAY;
26.[a000000157446009]=12,CHINA UNION PAY;
26.[a0000001574460]=12,CHINA UNION PAY;
26.[a000000333010101]=12,CHINA UNION PAY;
26.[a000000333010102]=12,CHINA UNION PAY;
26.[a000000333010103]=12,CHINA UNION PAY;
26.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
26.[1]=100;
26.[01]=100;
26.[+a0000000101030]=100;
26.[+a0000001570022]=100;
26.[a0000000031030]=100;
// M-CARD -----------------------------
26.[22]=101;
26.[a0000001570100]=101;
26.[a0000001574450]=101;
26.[a000000162010201]=101;
26.[m-card]=101;
// POSTFINANCE CARD -------------------
26.[21]=102;
26.[a0000001110101]=102;
26.[a0000001999999]=102;
26.[a0000001999998]=102;
26.[a0000001570051]=102;
26.[a0000001570050]=102;
26.[a0000001574445]=102;
26.[a0000001574446]=102;
26.[a0000001110505]=102,POSTCARD;
26.[a0000001570110]=143,PostFinance Mobile Coup.;
26.[a0000001570111]=144,PostFinance Mobile Paym.;
// BERN-CARD --------------------------
26.[42]=103;
26.[+a0000001570107]=103;
// ZUERI-CARD -------------------------
26.[41]=104;
26.[+a0000001570106]=104;
// BASLER-CARD ------------------------
26.[45]=105;
26.[+a0000001570108]=105;
// CASH -------------------------------
26.[+a0000001570023]=106;
// PHARMACARD --------------------------
26.[55]=107;
26.[+a0000001570105]=107;
// MOEVENPICK -------------------------
26.[46]=108;
26.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --------
26.[+a0000001570103]=110;
26.[+a000000157010B]=110;
26.[+a000000003000375]=110;
26.[81]=110;
// JELMOLI PAYCARD --------------------
26.[a0000001574462]=137;
// MANOR CARD (MYONE) -----------------
26.[84]=111;
26.[+a0000001570104]=111;
26.[+a0000001570109]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
26.[24]=116;
26.[a0000001574444]=116;
26.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
26.[26]=134;
26.[a0000001574458]=134;
26.[a00000000300037562]=134;
26.[a0000000037562]=134;
// POWERCARD (MFGroup)-----------------
26.[47]=117;
26.[+a000000157010D]=117;
26.[47]=117;
26.[a0000001574476]=117;
26.[a0000001574478]=117;
26.[a0000001574479]=117;
// WIR --------------------------------
26.[85]=118;
26.[+a000000157010C]=118;
// REKA  -------------------------------
26.[90]=120;
26.[d7560000011111]=120;
26.[+a000000157010E]=120;
26.[+d7560000010101]=120;
// REKA RAIL --------------------------
26.[45]=180;
26.[d7560000011112]=180;
26.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
26.[78]=178;
26.[d7560000011113]=178;
26.[d7560000010102]=178;
26.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
26.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
26.[+a000000157444E]=122,J-Geschenkkarte;
// ADUNO CARDS --------------
26.[+a0000001574448]=123,Aduno Kundenkarte;
26.[+a0000001574449]=124,Swiss Bonus Card;
26.[+a000000157444A]=125,ABO Card;
26.[a000000157010F]=142,ADUNO TEST;
// MEDIAMARKT --------------
26.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
26.[D7560000300101]=132,M-Budget;
// GIFT CARD (SIX)-----------------------
26.[a0000001574451]=133,Gift Card;
// SWISSCADEAU --------------------------
26.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
26.[a000000157445E]=140,BonGenie/Grieder Card; 
// GIFT CARD (BILLAG) -------------------
26.[a000000157010A]=141,GIFT CARD BILLAG;
// ACCARDA STAEDTEKARTE -----------------
26.[a0000001570112]=115,ACCARDA STAEDTEKARTE;
// KUNDENKARTE OFAC ---------------------
26.[a0000001570113]=145,Kundenkarte OFAC;
// REINVEST SELECT AG COMPANY CARD -----
26.[a0000001574447]=146,REINVEST SELECT;
// KUNDENKARTE PKZ -----------------------
26.[a000000157445D]=147,KUNDENKARTE PKZ; 
// JUBIN CARD ----------------------------
26.[a000000157445F]=148,JUBIN CARD;
// INNOCARD -------------------------------
26.[a0000001574448]=149,INNOCARD KUNDENKARTE;
26.[a000000157444A]=150,ABO CARD;
26.[a0000001574459]=151,SWISS HAPPY CARD;
26.[a0000001574461]=153,DATA STANDARDS;
26.[a0000001574463]=154,SWISSBONUS+;
// GE MONEY BANK ---------------------------
26.[a000000157444B]=156,GE MONEY BANK;
26.[a000000157444F]=157,BARRACUDA TEMPCARD;
// MERGER CARD -----------------------------
26.[a000000157444C]=158,MERGER CARD;
26.[a0000004840001]=158,MERGER CARD;
26.[a000000157444D]=159,MERGER CARD 2;
// BANCOMAT LUXEMBOURG -------------------
26.[a0000001574453]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ---------------------
26.[a0000001574454]=1001,AURORA LUXEMBOURG;
// BONCARD -------------------------------
26.[a0000001574455]=160,BONCARD POINTS;
26.[a000000157445B]=161,BONCARD PAY;
// FNAC ----------------------------------
26.[a0000001574456]=162,FNAC-KARTE;
// COMMIT CARD COMPANY CARD --------------
26.[a0000001574457]=163,COMMIT CARD;
// KUNDENKARTE LOEB ----------------------
26.[a000000157445C]=164,KUNDENKARTE LOEB;
// LUNCH-CHECK ---------------------------
26.[37]=172;
26.[a000000157447d]=172;
26.[d7560001150001]=172;
// CCV PAY -------------------------------
26.[35]=173;
26.[a000000157447b]=173;
// COOP EKZ ------------------------------
26.[37]=174;
26.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
26.[60]=175;
26.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
26.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
26.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
26.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
26.[+a0000001574480]=186;
// TWINT -------------------------------
26.[a000000157449e]=184,TWINT;
// SWISS PAY
26.[a0000008010001]=2000,SWISS PAY;
26.[37]=2000,SWISS PAY;
// SWISS PAY 2
26.[a00000015744BD]=2001,SWISS PAY 2;
26.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 27 (CH ADUNO BLADE):
// The card recognition is based on the ISSUER NUMBER and Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
27.[7]=1;
27.[07]=1;
27.[+a0000000043060]=1;
27.[+a0000001570021]=1;
// MASTERCARD -------------------------
27.[2]=2;
27.[02]=2;
27.[+a0000000041010]=2;
27.[+a0000001570020]=2;
// M-Budget
27.[a0000000000008]=2;
// JCB -------------------------------
27.[6]=3;
27.[06]=3;
27.[+a0000000651010]=3;
27.[+a0000001570040]=3;
// DINERS -----------------------------
27.[43]=4;
27.[+a0000001884443]=4;
27.[+a0000001574443]=4;
27.[+a0000001570060]=4;
27.[+a0000001523010]=4;
// VISA -------------------------------
27.[+a0000000031010]=5;
27.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
27.[+a0000000032010]=10;
27.[+a0000001570031]=10;
// V PAY -------------------------------
27.[96]=11,V PAY;
27.[a0000000032020]=11,V PAY;
27.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
27.[40]=6;
27.[+a00000002501]=6;
27.[+a0000001570010]=6;
27.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
27.[9]=12,CHINA UNION PAY;
27.[09]=12,CHINA UNION PAY;
27.[a000000157446003]=12,CHINA UNION PAY;
27.[a000000157446004]=12,CHINA UNION PAY;
27.[a000000157446005]=12,CHINA UNION PAY;
27.[a000000157446006]=12,CHINA UNION PAY;
27.[a000000157446009]=12,CHINA UNION PAY;
27.[a0000001574460]=12,CHINA UNION PAY;
27.[a000000333010101]=12,CHINA UNION PAY;
27.[a000000333010102]=12,CHINA UNION PAY;
27.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
27.[1]=100;
27.[01]=100;
27.[+a0000000101030]=100;
27.[+a0000001570022]=100;
27.[a0000000031030]=100;
// M-CARD -----------------------------
27.[22]=101;
27.[a0000001570100]=101;
27.[a0000001574450]=101;
27.[a000000162010201]=101;
27.[m-card]=101;
// POSTFINANCE CARD -------------------
27.[21]=102;
27.[a0000001110101]=102;
27.[a0000001999999]=102;
27.[a0000001999998]=102;
27.[a0000001570051]=102;
27.[a0000001570050]=102;
27.[a0000001574445]=102;
27.[a0000001574446]=102;
27.[a0000001570110]=143,PostFinance Mobile Coup.;
27.[a0000001570111]=144,PostFinance Mobile Paym.;
// BERN-CARD --------------------------
27.[42]=103;
27.[+a0000001570107]=103;
// ZUERI-CARD -------------------------
27.[41]=104;
27.[+a0000001570106]=104;
// BASLER-CARD ------------------------
27.[45]=105;
27.[+a0000001570108]=105;
// CASH -------------------------------
27.[+a0000001570023]=106;
// PHARMACARD --------------------------
27.[55]=107;
27.[+a0000001570105]=107;
// MOEVENPICK -------------------------
27.[46]=108;
27.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --------
27.[+a0000001570103]=110;
27.[+a000000157010B]=110;
27.[+a000000003000375]=110;
27.[81]=110;
// JELMOLI PAYCARD --------------------
27.[a0000001574462]=137;
// MANOR CARD (MYONE) -----------------
27.[84]=111;
27.[+a0000001570104]=111;
27.[+a0000001570109]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
27.[24]=116;
27.[a0000001574444]=116;
27.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
27.[26]=134;
27.[a0000001574458]=134;
27.[a00000000300037562]=134;
27.[a0000000037562]=134;
// POWERCARD (MFGroup)-----------------
27.[47]=117;
27.[+a000000157010D]=117;
27.[a0000001574476]=117;
27.[a0000001574478]=117;
27.[a0000001574479]=117;
// WIR --------------------------------
27.[85]=118;
27.[+a000000157010C]=118;
// REKA  -------------------------------
27.[90]=120;
27.[d7560000011111]=120;
27.[+a000000157010E]=120;
27.[+D7560000010101]=120;
// REKA RAIL --------------------------
27.[45]=180;
27.[d7560000011112]=180;
27.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
27.[78]=178;
27.[d7560000011113]=178;
27.[d7560000010102]=178;
27.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
27.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
27.[+a000000157444E]=122,J-Geschenkkarte;
// ADUNO CARDS --------------
27.[+a0000001574448]=123,Aduno Kundenkarte;
27.[+a0000001574449]=124,Swiss Bonus Card;
27.[+a000000157444A]=125,ABO Card;
27.[a000000157010F]=142,ADUNO TEST;
// MEDIAMARKT --------------
27.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
27.[D7560000300101]=132,M-Budget;
// GIFT CARD (SIX)-----------------------
27.[a0000001574451]=133,Gift Card;
// SWISSCADEAU --------------------------
27.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
27.[a000000157445E]=140,BonGenie/Grieder Card; 
// GIFT CARD (BILLAG) -------------------
27.[a000000157010A]=141,GIFT CARD BILLAG;
// ACCARDA STAEDTEKARTE -----------------
27.[a0000001570112]=115,ACCARDA STAEDTEKARTE;
// KUNDENKARTE OFAC ---------------------
27.[a0000001570113]=145,Kundenkarte OFAC;
// REINVEST SELECT AG COMPANY CARD -----
27.[a0000001574447]=146,REINVEST SELECT;
// KUNDENKARTE PKZ -----------------------
27.[a000000157445D]=147,KUNDENKARTE PKZ; 
// JUBIN CARD ----------------------------
27.[a000000157445F]=148,JUBIN CARD;
// INNOCARD -------------------------------
27.[a0000001574448]=149,INNOCARD KUNDENKARTE;
27.[a000000157444A]=150,ABO CARD;
27.[a0000001574459]=151,SWISS HAPPY CARD;
27.[a0000001574461]=153,DATA STANDARDS;
27.[a0000001574463]=154,SWISSBONUS+;
// GE MONEY BANK ---------------------------
27.[a000000157444B]=156,GE MONEY BANK;
27.[a000000157444F]=157,BARRACUDA TEMPCARD;
// MERGER CARD -----------------------------
27.[a000000157444C]=158,MERGER CARD;
27.[a0000004840001]=158,MERGER CARD;
27.[a000000157444D]=159,MERGER CARD 2;
// BANCOMAT LUXEMBOURG -------------------
27.[a0000001574453]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ---------------------
27.[a0000001574454]=1001,AURORA LUXEMBOURG;
// BONCARD -------------------------------
27.[a0000001574455]=160,BONARD POINTS;
27.[a000000157445B]=161,BONCARD PAY;
// FNAC ----------------------------------
27.[a0000001574456]=162,FNAC-KARTE;
// COMMIT CARD COMPANY CARD --------------
27.[a0000001574457]=163,COMMIT CARD;
// KUNDENKARTE LOEB ----------------------
27.[a000000157445C]=164,KUNDENKARTE LOEB;
// LUNCH-CHECK ---------------------------
27.[37]=172;
27.[a000000157447d]=172;
27.[d7560001150001]=172;
// CCV PAY -------------------------------
27.[35]=173;
27.[a000000157447b]=173;
// COOP EKZ ------------------------------
27.[37]=174;
27.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF --
27.[60]=175;
27.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
27.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
27.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
27.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
27. [+a0000001574480]=186;
// SWISS PAY
27.[a0000008010001]=2000,SWISS PAY;
27.[37]=2000,SWISS PAY;
// SWISS PAY 2
27.[a00000015744BD]=2001,SWISS PAY 2;
27.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 28 (NL BANKSYS / VIC Standard oriented C-ZAM/SMASH & C-ZAM/SPIN 1.06A (Draft)):
// The card recognition is based on mod_name (Bit 180).
// *******************************************************************************************************************************************
// VISA -------------------------------
28.[VISA]=5;
28.[Visa]=5;
28.[V Pay]=11;
// JCB --------------------------------
28.[JCB]=3;
// DINERS -----------------------------
28.[DINERS]=4;
28.[Diners]=4;
// AMEX -------------------------------
28.[AMEX]=6;
// MASTERCARD -------------------------
28.[ECMC]=2;
28.[MasterCard]=2;
// Maestro ----------------------------
28.[MAES]=1;
28.[Maestro]=1;
// Bancontact/Mister Cash -------------
28.[BANK]=700;
// Postbank ---------------------------
28.[GIRO]=9;
// EPWS --------------------------------
28.[EPWS]=702;
// CUP ---------------------------------
28.[CUP]=12;
// COMFORTCARD -------------------------
28.[CMFC]=704;
28.[COMF]=704;
// PRIMELINE----------------------------
28.[PRLI]=705;
28.[PRIM]=705;
// Basic Card --------------------------
28.[Basic Card]=706;
// Schoenenbon -------------------------
28.[Schoenenbon]=707;
// Fashion Card ------------------------
28.[Fashion Card]=708;
// CCV Card ----------------------------
28.[CCV Card]=709;
// Yourgift ----------------------------
28.[Yourgift H&G]=710;
28.[Yourgift G&M]=711;
28.[Yourgift M&M]=712;
28.[Yourgift B&H]=713;
28.[Yourgift H&T]=714;
28.[Yourgift F&F]=715;
28.[Yourgift W&D]=716;
28.[Yourgift G&C]=717;
// HORNBACH PROFICARD NL---------------
28.[HORN]=718;
28.[CCV]=718;
// Chipknip ----------------------------
28.[CHIP]=719;
//
// **********************************************************************************************************************************************
// Terminal type 29 (AT CARD COMPLETE INGENICO U32 / SMART / ZVT)
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
29.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
29.[5]=9,GIROCARD;
// EAPS -------------------------------
29.[7]=8,EAPS;
// Lastschrift ------
29.[9]=409,LASTSCHRIFT;
29.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
29.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
29.[13]=11,V PAY;
// BSW --------------------------------
29.[40]=404,BSW;
// MAESTRO ----------------------------
29.[46]=1;
// MASTERCARD -------------------------
29.[6]=2;
// DINERS -----------------------------
29.[12]=4;
// VISA -------------------------------
29.[10]=5;
// AMEX -------------------------------
29.[8]=6;
// JCB --------------------------------
29.[14]=3;
// MAESTRO DE / ELV -------------------
29.[2ELV]=410;
// MAESTRO DE / OLV -------------------
29.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
29.[2CASH]=412;
// MAESTRO DE / default ---------------
29.[2]=401;
// GELDKARTE --------------------------
29.[30]=402;
// IKEA FAMILY PAYCARD------------------
29.[76]=203,IKEA Family Paycard;
// HORNBACH Profi ---------------------
29.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
29.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
29.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
29.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
29.[42]=442,Comfort Card;
// Breuninger--------------------------
29.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
29.[+414]=414,AnniCard;
29.[+AnniCard]=414,AnniCard;
//
29.[212]=441,T-D1 XTRA;
29.[+T-D1 XTRA]=441,T-D1 XTRA;
29.[+T-D1]=441,T-D1 XTRA;
29.[+XTRA]=441,T-D1 XTRA;
//
29.[191]=479,Vodafone CallYa;
29.[+Vodafone CallYa]=479,Vodafone CallYa;
29.[+Vodafone]=479,Vodafone CallYa;
29.[+CallYa]=479,Vodafone CallYa;
//
29.[+80]=480,E-Plus Free&Easy;
29.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
29.[+E-Plus]=480,E-Plus Free&Easy;
29.[+Free&Easy]=480,E-Plus Free&Easy;
//
29.[188]=481,O2 LOOP;
29.[+O2 LOOP]=481,O2 LOOP;
29.[+O2]=481,O2 LOOP;
29.[+LOOP]=481,O2 LOOP;
//
29.[+431]=431,Premiere Flex;
29.[+Premiere Flex]=431,Premiere Flex;
29.[+Premiere]=431,Premiere Flex;
29.[+Flex]=431,Premiere Flex;
//
29.[189]=405,gobananas;
29.[+gobananas]=405,gobananas;
//
29.[+352]=452,Paysafecard;
29.[+Paysafecard]=452,Paysafecard;
//
// 218 OBB VORTEILSKARTE---------------
29.[228]=218;
// 219 SHOPIN-KARTE--------------------
29.[230]=219, SHOPIN-KARTE;
// 220 TLAPA-KARTE---------------------
29.[231]=220,TLAPA-KARTE; 
// AUSTRIA CARD -----------------------
29.[217]=217,Austria Card;
//
// SCHLECKER CARDS------------------
29.[807]=484,SCHLECKER PREPAID;
29.[249]=486,smobil;
29.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
29.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
29.[246]=488,FONIC;
29.[206]=489,BILDmobil;
29.[205]=490,Blau.de;
29.[207]=491,Congstar;
29.[475]=492,Lebara Mobile;
29.[248]=493,Simyo;
29.[884]=494,Lycamobile;
29.[917]=495,GT mobile;
// AVS -----------------------------
29.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
29.[+6010822]=497,Alnatura Gutscheinkarte;

// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
29.[+67]=1;
// COMFORT CARD -----------------------
29.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
29.[+5]=2;
29.[+222]=2;
29.[+223]=2;
29.[+224]=2;
29.[+225]=2;
29.[+226]=2;
29.[+227]=2;
29.[+228]=2;
29.[+229]=2;
29.[+230]=2;
29.[+231]=2;
29.[+232]=2;
29.[+233]=2;
29.[+234]=2;
29.[+235]=2;
29.[+236]=2;
29.[+237]=2;
29.[+238]=2;
29.[+239]=2;
29.[+240]=2;
29.[+241]=2;
29.[+242]=2;
29.[+243]=2;
29.[+244]=2;
29.[+245]=2;
29.[+246]=2;
29.[+247]=2;
29.[+248]=2;
29.[+249]=2;
29.[+250]=2;
29.[+251]=2;
29.[+252]=2;
29.[+253]=2;
29.[+254]=2;
29.[+255]=2;
29.[+256]=2;
29.[+257]=2;
29.[+258]=2;
29.[+259]=2;
29.[+261]=2;
29.[+262]=2;
29.[+263]=2;
29.[+264]=2;
29.[+265]=2;
29.[+266]=2;
29.[+267]=2;
29.[+268]=2;
29.[+269]=2;
29.[+270]=2;
29.[+271]=2;
29.[+272]=2;
// DINERS -----------------------------
29.[+36]=4;
// VISA -------------------------------
29.[+4]=5;
// AMEX -------------------------------
29.[+37]=6;
// JCB --------------------------------
29.[+35]=3;
// VALUEMASTER --------------------------------
29.[91]=222;
29.[9120032060009]=222;
// **********************************************************************************************************************************************
// Terminal type 30 (DE elPay)
// **********************************************************************************************************************************************
// MAESTRO DE / ELV -------------------
30.[1ELV]=410,ec National ELV;
// MAESTRO DE / ECC -------------------
30.[99]=412,ec-Cash Zahlung;
30.[4]=412,ec Kundenkarte;
// MAESTRO DE / OLV -------------------
30.[1]=411,ec National OLV;
// MAESTRO ----------------------------
30.[2]=1,ec International;
30.[3]=1,ec Ausland;
// VISA -------------------------------
30.[5]=5,Visa;
// MASTERCARD -------------------------
30.[6]=2,Eurocard;
// DINERS -----------------------------
30.[7]=4,Diners;
// AMEX -------------------------------
30.[8]=6,Amex;
// JCB --------------------------------
30.[9]=3,JCB;
// Breuninger -------------------------
30.[10]=438,Breuninger;
// IKEA FAMILY PLUS -------------------
30.[14]=403,IKEA Family Karte;
//
// **********************************************************************************************************************************************
// Terminal type 31 (IT INGENICO ABI Microcircuito ECR Protocol):
// The card recognition is primarily based on the ACQUIRER ID
// **********************************************************************************************************************************************
// Setefi -----------------------------
// MAESTRO -----------------------------
31.[00000000017]=1,MAESTRO;
31.[00000088001]=1,MAESTRO;
31.[00000000088]=1,MAESTRO;
31.[88105100066]=1,MAESTRO;
// PAGOBANCOMAT (DOMESTIC MAESTRO)-----
31.[00000000000]=801,PagoBancomat EMV;
31.[00000000015]=801,PagoBancomat;
31.[15]=801,PagoBancomat;
// CARTA BANCOMAT
31.[00000010002]=801;
// AMEX -------------------------------
31.[00000000002]=6,AMEX;
31.[88105000002]=6,AMEX;
31.[00000000019]=6,AMEX EMV;
// MASTERCARD -------------------------
31.[00000000003]=2,MASTERCARD;
31.[88105100004]=2,MASTERCARD;
31.[00000000078]=2,MASTERCARD;
31.[00000000018]=2,Mastercard_S;
// VISA -------------------------------
31.[00000000004]=5,VISA;
31.[00000000008]=5,VISA;
31.[88105100003]=5,VISA;
// DINERS -----------------------------
31.[00000000005]=4,DINERS;
// JCB --------------------------------
31.[00000000006]=3,JCB;
// AGOS -------------------------------
31.[00000000008]=804,AGOS;
31.[00000000015]=809,AGOS ITA;
31.[00000000021]=810,Agos Alsolia;
// FINDOMESTIC ------------------------
31.[00000000009]=805,findomestic;
31.[00000000020]=806,findomestic Gift;
31.[00000000022]=807,findomestic Aura;
31.[00000000023]=808,findomestic NOVA;
// FIDITALIA --------------------------
31.[00000000013]=811,Fiditalia;
31.[00000000074]=812,Fiditalia PROMOZIONI;
// MONETA Visa ------------------------
31.[00000000001]=813,MONETA VISA;
// MONETA MC --------------------------
31.[00000000016]=814,MONETA MC;
// CREDIRAMA --------------------------
31.[00000000007]=815,CREDIRAMA;
// ACCORD -----------------------------
31.[00000000010]=816,Accord;
// CLESSIDRA --------------------------
31.[00000000011]=817,Clessidra;
// LINEA ------------------------------
31.[00000000012]=818,Linea;
// TOYS	-------------------------------
31.[00000000014]=819,Toys;
// MAESTRO ----------------------------
// CHINA UNION PAY --------------------
31.[00000000027]=12,CHINA UNION PAY;
// Metro ------------------------------
31.[00000000029]=820,Metro;
// UNICREDIT --------------------------
// MASTERCARD -------------------------
31.[00000000070]=2,MASTERCARD;
31.[70]=2,MASTERCARD;
// MAESTRO ----------------------------
31.[00000000066]=1,MAESTRO;
31.[66]=1,MAESTRO;
// VISA -------------------------------
31.[00000000068]=5,VISA;
31.[68]=5,VISA;
// DINERS -----------------------------
31.[00000000005]=4,DINERS;
31.[5]=4,DINERS;
// AMEX -------------------------------
31.[00000000002]=6,AMERICAN EXPRESS;
31.[00000080006]=6,AMERICAN EXPRESS;
31.[2]=6,AMERICAN EXPRESS;
// JCB --------------------------------
31.[00000000067]=3,JCB;
31.[67]=3,JCB;
// PAGOBANCOMAT (DOMESTIC MAESTRO)-----
31.[00000000060]=801,MAESTRO;
31.[60]=801,MAESTRO;
31.[00000003226]=801,MAESTRO;
31.[3226]=801,MAESTRO;
31.[00000003111]=801,MAESTRO; 
31.[3111]=801,MAESTRO;
31.[00000002008]=801,MAESTRO;
31.[2008]=801,MAESTRO;
31.[00000003500]=801,MAESTRO;
31.[3500]=801,MAESTRO;
31.[88108100001]=801,MAESTRO;
// KEY CLIENT -------------------------
31.[00000000003]=802,KEY CLIENT;
31.[3]=802,KEY CLIENT;
// CARTASI ----------------------------
31.[00000000004]=803,CARTASI;
31.[4]=803,CARTASI;
31.[00000080006]=803,CARTASI;
31.[80006]=803,CARTASI;
// AGOS -------------------------------
31.[00000000006]=804,AGOS;
31.[6]=804,AGOS;
// FINDOMESTIC ------------------------
31.[00000000009]=805,FINDOMESTIC;
31.[9]=805,FINDOMESTIC;
// BANCO DI BRESCIA
31.[00000000046]=821,BANCO DI BRESCIA;
31.[46]=821,BANCO DI BRESCIA;
//
// **********************************************************************************************************************************************
// Terminal type 32 (ES INGENICO Terminal controlado via serie):
// The card recognition is primarily based on the CARD NUMBER PREFIX.
// **********************************************************************************************************************************************
// DINERS -----------------------------
32.[+300]=4,DINERS;
32.[+301]=4,DINERS;
32.[+302]=4,DINERS;
32.[+303]=4,DINERS;
32.[+304]=4,DINERS;
32.[+305]=4,DINERS;
32.[+3095]=4,DINERS;
32.[+3095]=4,DINERS;
32.[+3096]=4,DINERS;
32.[+3097]=4,DINERS;
32.[+3098]=4,DINERS;
32.[+3099]=4,DINERS;
32.[+36]=4,DINERS;
32.[+38]=4,DINERS;
32.[+39]=4,DINERS;
32.[+60110]=4,DINERS;
32.[+60112]=4,DINERS;
32.[+60113]=4,DINERS;
32.[+60114]=4,DINERS;
32.[+601174]=4,DINERS;
32.[+601177]=4,DINERS;
32.[+601178]=4,DINERS;
32.[+601179]=4,DINERS;
32.[+601186]=4,DINERS;
32.[+601187]=4,DINERS;
32.[+601188]=4,DINERS;
32.[+601189]=4,DINERS;
32.[+60119]=4,DINERS;
32.[+644]=4,DINERS;
32.[+645]=4,DINERS;
32.[+646]=4,DINERS;
32.[+647]=4,DINERS;
32.[+648]=4,DINERS;
32.[+649]=4,DINERS;
32.[+65]=4,DINERS;
// MASTERCARD -------------------------
32.[+500235]=2,MASTERCARD;
32.[+500766]=2,MASTERCARD;
32.[+501012]=2,MASTERCARD;
32.[+502029]=2,MASTERCARD;
32.[+503615]=2,MASTERCARD;
32.[+504507]=2,MASTERCARD;
32.[+504834]=2,MASTERCARD;
32.[+504837]=2,MASTERCARD;
32.[+5049]=2,MASTERCARD;
32.[+51]=2,MASTERCARD;
32.[+52]=2,MASTERCARD;
32.[+53]=2,MASTERCARD;
32.[+54,-545250]=2,MASTERCARD;
32.[+55]=2,MASTERCARD;
32.[+222]=2,MASTERCARD;
32.[+223]=2,MASTERCARD;
32.[+224]=2,MASTERCARD;
32.[+225]=2,MASTERCARD;
32.[+226]=2,MASTERCARD;
32.[+227]=2,MASTERCARD;
32.[+228]=2,MASTERCARD;
32.[+229]=2,MASTERCARD;
32.[+230]=2,MASTERCARD;
32.[+231]=2,MASTERCARD;
32.[+232]=2,MASTERCARD;
32.[+233]=2,MASTERCARD;
32.[+234]=2,MASTERCARD;
32.[+235]=2,MASTERCARD;
32.[+236]=2,MASTERCARD;
32.[+237]=2,MASTERCARD;
32.[+238]=2,MASTERCARD;
32.[+239]=2,MASTERCARD;
32.[+240]=2,MASTERCARD;
32.[+241]=2,MASTERCARD;
32.[+242]=2,MASTERCARD;
32.[+243]=2,MASTERCARD;
32.[+244]=2,MASTERCARD;
32.[+245]=2,MASTERCARD;
32.[+246]=2,MASTERCARD;
32.[+247]=2,MASTERCARD;
32.[+248]=2,MASTERCARD;
32.[+249]=2,MASTERCARD;
32.[+250]=2,MASTERCARD;
32.[+251]=2,MASTERCARD;
32.[+252]=2,MASTERCARD;
32.[+253]=2,MASTERCARD;
32.[+254]=2,MASTERCARD;
32.[+255]=2,MASTERCARD;
32.[+256]=2,MASTERCARD;
32.[+257]=2,MASTERCARD;
32.[+258]=2,MASTERCARD;
32.[+259]=2,MASTERCARD;
32.[+261]=2,MASTERCARD;
32.[+262]=2,MASTERCARD;
32.[+263]=2,MASTERCARD;
32.[+264]=2,MASTERCARD;
32.[+265]=2,MASTERCARD;
32.[+266]=2,MASTERCARD;
32.[+267]=2,MASTERCARD;
32.[+268]=2,MASTERCARD;
32.[+269]=2,MASTERCARD;
32.[+270]=2,MASTERCARD;
32.[+271]=2,MASTERCARD;
32.[+272]=2,MASTERCARD;
// CHINA UNION PAY --------------------
32.[+456351]=12,CHINA UNION PAY;
32.[+602969]=12,CHINA UNION PAY;
32.[+603367]=12,CHINA UNION PAY;
32.[+620]=12,CHINA UNION PAY;
32.[+621]=12,CHINA UNION PAY;
32.[+622]=12,CHINA UNION PAY;
32.[+623]=12,CHINA UNION PAY;
32.[+624]=12,CHINA UNION PAY;
32.[+625]=12,CHINA UNION PAY;
// MAESTRO ----------------------------
32.[+50]=1,MAESTRO;
32.[+54250]=1,MAESTRO;
32.[+56]=1,MAESTRO;
32.[+57]=1,MAESTRO;
32.[+58]=1,MAESTRO;
32.[+59]=1,MAESTRO;
32.[+60]=1,MAESTRO;
32.[+61]=1,MAESTRO;
32.[+62]=1,MAESTRO;
32.[+63]=1,MAESTRO;
32.[+64]=1,MAESTRO;
32.[+66]=1,MAESTRO;
32.[+67]=1,MAESTRO;
32.[+68]=1,MAESTRO;
32.[+69]=1,MAESTRO;
// VISA ELECTRON -----------------------
32.[+400115]=10,VISA ELECTRON;
32.[+400837]=10,VISA ELECTRON;
32.[+400838]=10,VISA ELECTRON;
32.[+400839]=10,VISA ELECTRON;
32.[+412921]=10,VISA ELECTRON;
32.[+412922]=10,VISA ELECTRON;
32.[+412923]=10,VISA ELECTRON;
32.[+417935]=10,VISA ELECTRON;
32.[+419740]=10,VISA ELECTRON;
32.[+419741]=10,VISA ELECTRON;
32.[+419773]=10,VISA ELECTRON;
32.[+419774]=10,VISA ELECTRON;
32.[+419775]=10,VISA ELECTRON;
32.[+424519]=10,VISA ELECTRON;
32.[+424962]=10,VISA ELECTRON;
32.[+424963]=10,VISA ELECTRON;
32.[+437860]=10,VISA ELECTRON;
32.[+444000]=10,VISA ELECTRON;
32.[+459472]=10,VISA ELECTRON;
32.[+484406]=10,VISA ELECTRON;
32.[+484407]=10,VISA ELECTRON;
32.[+484408]=10,VISA ELECTRON;
32.[+484409]=10,VISA ELECTRON;
32.[+484410]=10,VISA ELECTRON;
32.[+484411]=10,VISA ELECTRON;
32.[+484413]=10,VISA ELECTRON;
32.[+484414]=10,VISA ELECTRON;
32.[+484418]=10,VISA ELECTRON;
32.[+484419]=10,VISA ELECTRON;
32.[+484420]=10,VISA ELECTRON;
32.[+484421]=10,VISA ELECTRON;
32.[+484422]=10,VISA ELECTRON;
32.[+484423]=10,VISA ELECTRON;
32.[+484424]=10,VISA ELECTRON;
32.[+484425]=10,VISA ELECTRON;
32.[+484426]=10,VISA ELECTRON;
32.[+484428]=10,VISA ELECTRON;
32.[+484429]=10,VISA ELECTRON;
32.[+48443]=10,VISA ELECTRON;
32.[+48444]=10,VISA ELECTRON;
32.[+484450]=10,VISA ELECTRON;
32.[+484451]=10,VISA ELECTRON;
32.[+484452]=10,VISA ELECTRON;
32.[+484453]=10,VISA ELECTRON;
32.[+484454]=10,VISA ELECTRON;
32.[+484455]=10,VISA ELECTRON;
32.[+49173]=10,VISA ELECTRON;
32.[+49174]=10,VISA ELECTRON;
32.[+49175]=10,VISA ELECTRON;
// VISA -------------------------------
32.[+4]=5,VISA;
// AMEX -------------------------------
32.[+34]=6,AMERICAN EXPRESS;
32.[+37]=6,AMERICAN EXPRESS;
// JCB --------------------------------
32.[+35]=3,JCB; 
//
// **********************************************************************************************************************************************
// Terminal type 33 (PL First Data / Polcard ECR-POS Interface):
// The card recognition is not possible. This protocol will always return "CARD UNKNOWN"
// **********************************************************************************************************************************************
//
// **********************************************************************************************************************************************
// Terminal type 34 (AT PAYLIFE):
// The card recognition is based on the CARD NAME, given by the interface.
// (As default field CCC from p_belegzeile_1 or m_belegzeile_1, (supported by terminal software version 1.64 and greater) 
//  else m_name or, in the case of debit cards, the 3 first characters of p_text
//  else by card name)
// **********************************************************************************************************************************************
// MAESTRO -----------------------------
34.[d01]=1;
34.[d02]=1;
34.[edc]=1;
// MASTERCARD --------------------------
34.[d05]=2;
34.[c01]=2;
34.[c02]=2;
34.[KMC]=2;
34.[+mastercard]=2;
34.[+mc ]=2;
// JCB --------------------------------
34.[KJB]=3;
34.[d11]=3;
34.[c06]=3;
34.[+jcb]=3;
// DINERS -----------------------------
34.[d10]=4;
34.[c05]=4;
34.[KDI]=4;
34.[+diners]=4;
// VISA ELECTRON-----------------------
34.[+visa elec]=10;
// V PAY-------------------------------
34.[d07]=11;
34.[+v pay]=11;
34.[KVP]=11;
// VISA -------------------------------
34.[KVI]=5;
34.[d08]=5;
34.[c03]=5;
34.[+visa]=5;
34.[citibank visa]=5;
34.[hanseatic bank]=5;
34.[anzsmartpay]=5;
34.[ykb worldcard]=5;
// AMEX -------------------------------
34.[KAM]=6;
34.[d09]=6;
34.[c04]=6;
34.[+american]=6;
34.[+amex]=6;
// QUICK PIN based---------------------
34.[d06]=200;
34.[QCK]=200;
// BANKOMAT (MAESTRO AT) PIN based-----
34.[B-K]=201;
// BANKOMAT PIN based (not guaranteed)-
34.[BKN]=201;
34.[bkn]=201;
// JAPAN AIRLINES ---------------------
34.[+japan]=7;
// Euro Alliance of Payment Schemes----
34.[+eaps]=8;
// CUP (China UnionPay)----------------
34.[d12]=12;
34.[c07]=12;
34.[+cup]=12;
34.[+KCU]=12;
34.[+china]=12;
// FLOTTENKARTE (TYP 4)
34.[TYP4]=204,Flottenkarte;
// LOKALE KARTE (TYP 3)
34.[TYP3]=205;
// Inside Dornbirn
34.[V01OLP]=221,Inside Dornbirn;
34.[V01]=221,Inside Dornbirn;
34.[+Inside Dornbirn]=221,Inside Dornbirn;
//
// **********************************************************************************************************************************************
// Terminal type 35 (CH CCV JERONIMO Vx810 with CCV JIL):
// The card recognition is based on the ISSUER NUMBER and Application-ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
35.[7]=1;
35.[07]=1;
35.[a0000000043060]=1;
35.[a0000001570021]=1;
// MASTERCARD -------------------------
35.[2]=2;
35.[02]=2;
35.[+a0000000041010]=2;
35.[a0000001570020]=2;
// M-Budget MasterCard
35.[25]=2;
35.[a0000000000008]=2;
// JCB --------------------------------
35.[6]=3;
35.[06]=3;
35.[a0000000651010]=3;
35.[a0000001570040]=3;
// DINERS -----------------------------
35.[43]=4;
35.[+a0000001884443]=4;
35.[+a0000001574443]=4;
35.[+a0000001570060]=4;
35.[+a0000001523010]=4;
// VISA -------------------------------
35.[5]=5;
35.[05]=5;
35.[+a0000000031010]=5;
35.[+a0000001570030]=5;
35.[d0000001570031]=5;
// VISA ELECTRON-----------------------
35.[+a0000000032010]=10;
35.[+a0000001570031]=10;
// V PAY -------------------------------
35.[+a0000000032020]=11;
35.[a0000001574452]=11;
// AMEX -------------------------------
35.[40]=6;
35.[a000000025]=6;
35.[a0000001570010]=6;
35.[a00000002501]=6;
35.[a0000000291010]=6;
35.[a000000025010102]=6;
35.[a000000025010104]=6;
35.[a000000025010402]=6;
35.[a000000025010501]=6;
35.[a000000025010801]=6;
35.[a000000025010902]=6;
// CHINA UNION PAY --------------------
35.[9]=12,CHINA UNION PAY;
35.[09]=12,CHINA UNION PAY;
35.[a000000157446003]=12,CHINA UNION PAY;
35.[a000000157446004]=12,CHINA UNION PAY;
35.[a000000157446005]=12,CHINA UNION PAY;
35.[a000000157446006]=12,CHINA UNION PAY;
35.[a000000157446009]=12,CHINA UNION PAY;
35.[a0000001574460]=12,CHINA UNION PAY;
35.[a000000333010101]=12,CHINA UNION PAY;
35.[a000000333010102]=12,CHINA UNION PAY;
35.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
35.[1]=100;
35.[01]=100;
35.[a0000000101030]=100;
35.[a0000001570022]=100;
35.[a0000000031030]=100;
// M-CARD -----------------------------
35.[22]=101;
35.[+a0000001570100]=101;
// POSTCARD ---------------------------
35.[21]=102;
35.[a0000001110101]=102;
35.[a0000001110102]=102;
35.[a0000001110103]=102;
35.[+a0000001999999]=102;
35.[+a0000001999998]=102;
35.[+a0000001570050]=102;
35.[+a0000001570051]=102;
35.[+a0000001574445]=102;
35.[+a0000001574446]=102;
35.[a0000001110201]=102;
35.[a0000001110202]=102;
35.[a0000001110505]=102,POSTCARD;
35.[a0000001110110]=102,POSTCARD;
35.[a0000001110510]=102,POSTCARD;
35.[a0000001110210]=102,POSTCARD;
// BERN CARD --------------------------
35.[+a0000001570107]=103;
// ZUERI CARD -------------------------
35.[+a0000001570106]=104;
// BASLER CARD ------------------------
35.[+a0000001570108]=105;
// CASH -------------------------------
35.[+a0000001570023]=106;
// PHARMACARD --------------------------
35.[+a0000001570105]=107;
// MOEVENPICK -------------------------
35.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --------
// JELMOLI PAYCARD --------------------
35.[a0000001574462]=137;
35.[68]=110;
35.[81]=110;
35.[+a0000001570103]=110;
35.[+a000000157010B]=110;
35.[+a000000003000375]=110;
// MY ONE -----------------------------
35.[+a0000001570104]=111;
// ORANGE -----------------------------
35.[75]=112;
// Sunrise ----------------------------
35.[76]=113;
// SWISSCOM ---------------------------
35.[77]=114;
// JELMOLI SHOPPING BONUS CARD --------
35.[81]=110;
35.[a00000000300037561]=110;
// JELMOLI PAYCARD --------------------
35.[a0000001574462]=137;
// COOP SUPERCARD PLUS MASTERCARD-------
35.[24]=116;
35.[a0000001574444]=116;
35.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
35.[26]=134;
35.[a0000001574458]=134;
35.[a00000000300037562]=134;
35.[a0000000037562]=134;
// REINVEST SELECT AG COMPANY CARD -----
35.[a0000001574447]=146,REINVEST SELECT;
// POWER CARD -------------------------
35.[+a000000157010D]=117;
35.[47]=117;
35.[a0000001574476]=117;
35.[a0000001574478]=117;
35.[a0000001574479]=117;
// WIR --------------------------------
35.[50]=118;
35.[+a000000157010C]=118;
// REKA  -------------------------------
35.[90]=120;
35.[d7560000011111]=120;
35.[+a000000157010E]=120;
35.[d7560000010101]=120;
35.[a0000001880101]=120;
35.[a0000001574464]=120;
// REKA RAIL --------------------------
35.[45]=180;
35.[d7560000011112]=180;
35.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
35.[78]=178;
35.[d7560000011113]=178;
35.[d7560000010102]=178;
35.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
35.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
35.[+a000000157444E]=122,J-Geschenkkarte;
// COMMTRAIN/ADUNO CARDS --------------
35.[+a0000001574448]=123,Aduno Kundenkarte;
35.[+a0000001574449]=124,Swiss Bonus Card;
35.[+a000000157444A]=125,ABO Card;
// MEDIAMARKT --------------
35.[+a0000001570109]=126,Mediamarkt Card;
// YALLO ------------------------------
35.[73]=127,Yallo;
// COOP MOBILE ------------------------
35.[78]=128,Coop Mobile;
// PHONECARD ---------------------------
35.[79]=129,phoneCard;
// ORANGECLICK.CH ----------------------
35.[80]=130,orangeclick.ch;
// M-BUDGET MOBILE ---------------------
35.[82]=131,M-Budget Mobile;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
35.[D7560000300101]=132,M-Budget;
// GIFT CARD (SIX)-----------------------
35.[a0000001574451]=133,Gift Card;
// SWISSCADEAU --------------------------
35.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
35.[a000000157445E]=140,BonGenie/Grieder Card; 
// GIFT CARD (BILLAG) -------------------
35.[a000000157010A]=141,GIFT CARD BILLAG;
// ACCARDA STAEDTEKARTE -----------------
35.[a0000001570112]=115,ACCARDA STAEDTEKARTE;
// KUNDENKARTE OFAC ---------------------
35.[a0000001570113]=145,Kundenkarte OFAC;
// REINVEST SELECT AG COMPANY CARD -----
35.[a0000001574447]=146,REINVEST SELECT;
// KUNDENKARTE PKZ -----------------------
35.[a000000157445D]=147,KUNDENKARTE PKZ; 
// JUBIN CARD ----------------------------
35.[a000000157445F]=148,JUBIN CARD;
// INNOCARD -------------------------------
35.[a0000001574448]=149,INNOCARD KUNDENKARTE;
35.[a000000157444A]=150,ABO CARD;
35.[a0000001574459]=151,SWISS HAPPY CARD;
35.[a0000001574461]=153,DATA STANDARDS;
35.[a0000001574463]=154,SWISSBONUS+;
// MERGER CARD -----------------------------
35.[a000000157444C]=158,MERGER CARD;
35.[a0000004840001]=158,MERGER CARD;
35.[a000000157444D]=159,MERGER CARD 2;
// BONCARD ------------------------------
35.[a0000001574455]=160,BONARD POINTS;
35.[a000000157445B]=161,BONCARD PAY;
35.[66]=160,BONARD POINTS;
35.[67]=161,BONCARD PAY;
// BANCOMAT LUXEMBOURG ------------------
35.[+D44200002542414E]=1000,BANCOMAT LUXEMBOURG;
// LUNCH-CHECK --------------------------
35.[37]=172;
35.[a000000157447d]=172;
35.[d7560001150001]=172;
// CCV PAY ------------------------------
35.[35]=173;
35.[a000000157447b]=173;
// COOP EKZ -----------------------------
35.[37]=174;
35.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
35.[60]=175;
35.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
35.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
35.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
35.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
35.[+a0000001574480]=186;
// SWISS PAY
35.[a0000008010001]=2000,SWISS PAY;
35.[37]=2000,SWISS PAY;
// SWISS PAY 2
35.[a00000015744BD]=2001,SWISS PAY 2;
35.[37]=2001,SWISS PAY 2;
// TWINT -------------------------------
35.[a000000157449e]=184,TWINT;
// Alipay ---------------------
35.[a00000015744A0]=17,AliPay;
// WECHAT PAY -----------------------
35.[a00000015744C6]=18,WeChat Pay;
//
// **********************************************************************************************************************************************
// Terminal type 36 (FR/ES AXIS C3):

// The card recognition is based on the cCardType (FFU: in combination with SSCard); fallback recognition by PAN  
// **********************************************************************************************************************************************
// AMEX ---------------------------------
36.[6]=6,AMERICAN EXPRESS;
// China Union Pay Natixis/China Union Pay 
36.[m]=12,CHINA UNION PAY;
// Diners Club de France ----------------
36.[7]=4,DINERS CLUB FRANCE;
// JCB piste/JCB EMV --------------------
36.[D]=3,JCB;
36.[+35]=3,JCB;
// MASTERCARD (by PAN)------------------
36.[+50]=2,MASTERCARD;
36.[+51]=2,MASTERCARD;
36.[+52]=2,MASTERCARD;
36.[+53]=2,MASTERCARD;
36.[+54]=2,MASTERCARD;
36.[+55]=2,MASTERCARD;
36.[+222]=2,MASTERCARD;
36.[+223]=2,MASTERCARD;
36.[+224]=2,MASTERCARD;
36.[+225]=2,MASTERCARD;
36.[+226]=2,MASTERCARD;
36.[+227]=2,MASTERCARD;
36.[+228]=2,MASTERCARD;
36.[+229]=2,MASTERCARD;
36.[+230]=2,MASTERCARD;
36.[+231]=2,MASTERCARD;
36.[+232]=2,MASTERCARD;
36.[+233]=2,MASTERCARD;
36.[+234]=2,MASTERCARD;
36.[+235]=2,MASTERCARD;
36.[+236]=2,MASTERCARD;
36.[+237]=2,MASTERCARD;
36.[+238]=2,MASTERCARD;
36.[+239]=2,MASTERCARD;
36.[+240]=2,MASTERCARD;
36.[+241]=2,MASTERCARD;
36.[+242]=2,MASTERCARD;
36.[+243]=2,MASTERCARD;
36.[+244]=2,MASTERCARD;
36.[+245]=2,MASTERCARD;
36.[+246]=2,MASTERCARD;
36.[+247]=2,MASTERCARD;
36.[+248]=2,MASTERCARD;
36.[+249]=2,MASTERCARD;
36.[+250]=2,MASTERCARD;
36.[+251]=2,MASTERCARD;
36.[+252]=2,MASTERCARD;
36.[+253]=2,MASTERCARD;
36.[+254]=2,MASTERCARD;
36.[+255]=2,MASTERCARD;
36.[+256]=2,MASTERCARD;
36.[+257]=2,MASTERCARD;
36.[+258]=2,MASTERCARD;
36.[+259]=2,MASTERCARD;
36.[+261]=2,MASTERCARD;
36.[+262]=2,MASTERCARD;
36.[+263]=2,MASTERCARD;
36.[+264]=2,MASTERCARD;
36.[+265]=2,MASTERCARD;
36.[+266]=2,MASTERCARD;
36.[+267]=2,MASTERCARD;
36.[+268]=2,MASTERCARD;
36.[+269]=2,MASTERCARD;
36.[+270]=2,MASTERCARD;
36.[+271]=2,MASTERCARD;
36.[+272]=2,MASTERCARD;
// MAESTRO (by PAN)---------------------
36.[+67]=1,MAESTRO;
36.[+63]=1,MAESTRO;
36.[+56]=1,MAESTRO;
36.[+57]=1,MAESTRO;
36.[+58]=1,MAESTRO;
36.[+59]=1,MAESTRO;
// VISA (by PAN)-----------------------
//36.[+4970]=900,CARTE BLEUE;
//36.[+4971]=900,CARTE BLEUE;
//36.[+4973]=900,CARTE BLEUE;
//36.[+4974]=900,CARTE BLEUE;
//36.[+4975]=900,CARTE BLEUE;
//36.[+4976]=900,CARTE BLEUE;
//36.[+4978]=900,CARTE BLEUE;
36.[+4]=5,VISA;
// ACCORD ------------------------------
36.[p]=901,ACCORD EMV;
36.[V]=901,ACCORD Piste; 
36.[w]=901,ACCORD sans contact;
// Alphyra (Payzone) --------------------
36.[k]=902,ALPHYRA;
// Cafeteria Casino ---------------------
36.[Q]=903,CAFETERIA CASINO;
// Auchan Telecom -----------------------
36.[u]=904,AUCHAN TELECOM;
// PNF Credit Mutuel --------------------
36.[W]=905,PNF CREDIT MUTUEL;
// Employe ED ---------------------------
36.[r]=906,EMPLOYE ED;
// EVEREST -------------------------------
36.[5]=907,EVEREST;
// COFINOGA -----------------------------
36.[l]=908,COFINOGA EMV; 
36.[2]=908,COFINOGA Piste et Sans Contact;
// Cheque -------------------------------
36.[M]=909,CHEQUE;
36.[N]=909,CHEQUE;
36.[O]=909,CHEQUE;
36.[P]=909,CHEQUE;
// Caledonienne de Service Bancaire ------
36.[d]=910,CALEDONIENNE DE SERVICE BANCAIRE;
// Reflexe/Aurore - protocole CETNCB CETDIS-
36.[8]=911,AURORE;
// Datavantage (GIFTCARD) ------------------
36.[i]=912,DATAVANTAGE;
// Edel - Leclerc --------------------------
36.[H]=913,EDEL;
// Total - parking -------------------------
36.[X]=914,ELF;
// Pluriel - FranFinance -------------------
36.[9]=915,FFI;
// Fnac ------------------------------------
36.[g]=916,FNAC; 
// Finaref ---------------------------------
36.[F]=917,FINAREF; 
// Gestion de Avoirs -----------------------
36.[z]=918,GAV; 
// Galerie Lafayette -----------------------
36.[c]=919,GLF; 
// Magellan --------------------------------
36.[y]=920,MGL;
// Multissime ------------------------------
36.[U]=921,MULTISSIME;
// PNF Optalion - credit Lyonnais ----------
36.[h]=922,OPTALION;
// Pass piste/Pass EMV ---------------------
36.[3]=923,PASS;
// Pegasus ---------------------------------
36.[v]=924,PGS;
// Moneo (Porte Monnaie Electronique) ------
36.[S]=925,PME;
// Promostim (Record Card) -----------------
36.[a]=926,PROMOSTIM;
// Prosodie --------------------------------
36.[j]=927,PROSODIE;
// Sofinco ---------------------------------
36.[B]=928,SOFINCO;
// Sodelem - carte petrole -----------------
36.[q]=929,SODELEM;
// Smiles ----------------------------------
36.[Y]=930,SMILES;
//
// *********************************************************************************************************************************************
// Terminal type 37 (CH INNOCARD ARTEMA DESK / ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
37.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
37.[5]=9,GIROCARD;
// EAPS -------------------------------
37.[7]=8,EAPS;
// Lastschrift ------
37.[9]=409,LASTSCHRIFT;
37.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
37.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
37.[13]=11,V PAY;
// BSW --------------------------------
37.[40]=404,BSW;
// POSTCARD ----------------------------
37.[69]=102,POSTFINANCE CARD;
37.[236]=102,POSTFINANCE CARD;
37.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
37.[46]=1;
// MASTERCARD -------------------------
37.[6]=2;
// DINERS -----------------------------
37.[12]=4;
// VISA -------------------------------
37.[10]=5;
// AMEX -------------------------------
37.[8]=6;
// JCB --------------------------------
37.[14]=3;
// HORNBACH Profi ---------------------
37.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
37.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
37.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
37.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
37.[42]=442,Comfort Card;
// Breuninger--------------------------
37.[38]=438,Breuninger;
37.[+352]=452,Paysafecard;
37.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
37.[23]=160,boncard POINTS;
37.[25]=161,boncard PAY;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
37.[218]=175;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
37.[+67]=1;
// COMFORT CARD -----------------------
37.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
37.[+5]=2;
37.[+222]=2;
37.[+223]=2;
37.[+224]=2;
37.[+225]=2;
37.[+226]=2;
37.[+227]=2;
37.[+228]=2;
37.[+229]=2;
37.[+230]=2;
37.[+231]=2;
37.[+232]=2;
37.[+233]=2;
37.[+234]=2;
37.[+235]=2;
37.[+236]=2;
37.[+237]=2;
37.[+238]=2;
37.[+239]=2;
37.[+240]=2;
37.[+241]=2;
37.[+242]=2;
37.[+243]=2;
37.[+244]=2;
37.[+245]=2;
37.[+246]=2;
37.[+247]=2;
37.[+248]=2;
37.[+249]=2;
37.[+250]=2;
37.[+251]=2;
37.[+252]=2;
37.[+253]=2;
37.[+254]=2;
37.[+255]=2;
37.[+256]=2;
37.[+257]=2;
37.[+258]=2;
37.[+259]=2;
37.[+261]=2;
37.[+262]=2;
37.[+263]=2;
37.[+264]=2;
37.[+265]=2;
37.[+266]=2;
37.[+267]=2;
37.[+268]=2;
37.[+269]=2;
37.[+270]=2;
37.[+271]=2;
37.[+272]=2;
// DINERS -----------------------------
37.[+36]=4;
// VISA -------------------------------
37.[+4]=5;
// AMEX -------------------------------
37.[+37]=6;
// JCB --------------------------------
37.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
37.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  ------------------------------
37.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
37.[+a0000001574480]=186;
//
// *********************************************************************************************************************************************
// Terminal type 38 (CH INNOCARD OPTIMUM FAMILY / ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *********************************************************************************************************************************************
// girocard ---------------------------
38.[5]=9,GIROCARD;
// EAPS -------------------------------
38.[7]=8,EAPS;
// Lastschrift ------
38.[9]=409,LASTSCHRIFT;
38.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
38.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
38.[13]=11,V PAY;
// BSW --------------------------------
38.[40]=404,BSW;
// POSTCARD ----------------------------
38.[69]=102,POSTFINANCE CARD;
38.[236]=102,POSTFINANCE CARD;
38.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
38.[46]=1;
// MASTERCARD -------------------------
38.[6]=2;
// DINERS -----------------------------
38.[12]=4;
// VISA -------------------------------
38.[10]=5;
// AMEX -------------------------------
38.[8]=6;
// JCB --------------------------------
38.[14]=3;
// HORNBACH Profi ---------------------
38.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
38.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
38.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
38.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
38.[42]=442,Comfort Card;
// Breuninger--------------------------
38.[38]=438,Breuninger;
38.[+352]=452,Paysafecard;
38.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
38.[23]=160,boncard POINTS;
38.[25]=161,boncard PAY;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
38.[218]=175;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
38.[+67]=1;
// COMFORT CARD -----------------------
38.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
38.[+5]=2;
38.[+222]=2;
38.[+223]=2;
38.[+224]=2;
38.[+225]=2;
38.[+226]=2;
38.[+227]=2;
38.[+228]=2;
38.[+229]=2;
38.[+230]=2;
38.[+231]=2;
38.[+232]=2;
38.[+233]=2;
38.[+234]=2;
38.[+235]=2;
38.[+236]=2;
38.[+237]=2;
38.[+238]=2;
38.[+239]=2;
38.[+240]=2;
38.[+241]=2;
38.[+242]=2;
38.[+243]=2;
38.[+244]=2;
38.[+245]=2;
38.[+246]=2;
38.[+247]=2;
38.[+248]=2;
38.[+249]=2;
38.[+250]=2;
38.[+251]=2;
38.[+252]=2;
38.[+253]=2;
38.[+254]=2;
38.[+255]=2;
38.[+256]=2;
38.[+257]=2;
38.[+258]=2;
38.[+259]=2;
38.[+261]=2;
38.[+262]=2;
38.[+263]=2;
38.[+264]=2;
38.[+265]=2;
38.[+266]=2;
38.[+267]=2;
38.[+268]=2;
38.[+269]=2;
38.[+270]=2;
38.[+271]=2;
38.[+272]=2;
// DINERS -----------------------------
38.[+36]=4;
// VISA -------------------------------
38.[+4]=5;
// AMEX -------------------------------
38.[+37]=6;
// JCB --------------------------------
38.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
38.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  -------------------------------
38.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
38.[+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 39 (DE PAYSYS):
// The card recognition is based on ISSUER NUMBER, CARD NUMBER or APPLICATION ID.
// **********************************************************************************************************************************************
// CHINA UNION PAY --------------------
39.[9]=12,CHINA UNION PAY;
39.[09]=12,CHINA UNION PAY;
39.[a000000157446003]=12,CHINA UNION PAY;
39.[a000000157446004]=12,CHINA UNION PAY;
39.[a000000157446005]=12,CHINA UNION PAY;
39.[a000000157446006]=12,CHINA UNION PAY;
39.[a000000157446009]=12,CHINA UNION PAY;
39.[a0000001574460]=12,CHINA UNION PAY;
39.[a000000333010101]=12,CHINA UNION PAY;
39.[a000000333010102]=12,CHINA UNION PAY;
39.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO ----------------------------
39.[7]=1;
39.[07]=1;
// MASTERCARD -------------------------
39.[2]=2;
39.[02]=2;
// JCB --------------------------------
39.[6]=3;
39.[06]=3;
// DINERS -----------------------------
39.[43]=4;
// VISA -------------------------------
39.[5]=5;
39.[05]=5;
// V PAY -------------------------------
39.[96]=11,V PAY;
// AMEX -------------------------------
39.[40]=6;
// MAESTRO CH -------------------------
39.[1]=100;
39.[01]=100;
// M-CARD -----------------------------
39.[22]=101;
// POSTCARD ---------------------------
39.[21]=102;
// BERN CARD --------------------------
39.[41]=103;
39.[42]=103;
39.[45]=103;
// MOEVENPICK -------------------------
39.[46]=108;
// LOKALKARTE (READ OVER USB-PORT)
39.[80]=135;
// JELMOLI SHOPPING BONUS CARD --------
39.[81]=110;
// MYONE ------------------------------
39.[84]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
39.[24]=116;
// COOP SUPERCARD PLUS VISA-------------
39.[26]=134;
// REKA  -------------------------------
39.[90]=120;
39.[d7560000011111]=120;
39.[+a000000157010E]=120;
39.[d7560000010101]=120;
39.[a0000001880101]=120;
39.[a0000001574464]=120;
// REKA RAIL --------------------------
39.[45]=180;
39.[d7560000011112]=180;
39.[+a0000001574495]=180; 
// REKA LUNCH CARD --------------------
39.[78]=178;
39.[d7560000011113]=178;
39.[d7560000010102]=178;
39.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
39.[a000000157448C]=181;
// POWERCARD --------------------------
39.[47]=117;
39.[a0000001574476]=117;
39.[a0000001574478]=117;
39.[a0000001574479]=117;
// BONCARD ----------------------------
39.[66]=160,BONARD POINTS;
39.[67]=161,BONCARD PAY;
// LUNCH-CHECK ------------------------
39.[37]=172;
39.[a000000157447d]=172;
39.[d7560001150001]=172;
// CCV PAY ----------------------------
39.[35]=173;
// COOP EKZ ---------------------------
39.[37]=174;
39.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
39.[60]=175;
// CO2NEUTRAL PAY -----------------------
39.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
39.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
39.[a0000001574486]=179;
//if recognition by Issuer Number fails, try by card number prefix
// MASTERCARD -------------------------
39.[+50]=2;
39.[+51]=2;
39.[+52]=2;
39.[+53]=2
39.[+54]=2;
39.[+55]=2;
39.[+222]=2;
39.[+223]=2;
39.[+224]=2;
39.[+225]=2;
39.[+226]=2;
39.[+227]=2;
39.[+228]=2;
39.[+229]=2;
39.[+230]=2;
39.[+231]=2;
39.[+232]=2;
39.[+233]=2;
39.[+234]=2;
39.[+235]=2;
39.[+236]=2;
39.[+237]=2;
39.[+238]=2;
39.[+239]=2;
39.[+240]=2;
39.[+241]=2;
39.[+242]=2;
39.[+243]=2;
39.[+244]=2;
39.[+245]=2;
39.[+246]=2;
39.[+247]=2;
39.[+248]=2;
39.[+249]=2;
39.[+250]=2;
39.[+251]=2;
39.[+252]=2;
39.[+253]=2;
39.[+254]=2;
39.[+255]=2;
39.[+256]=2;
39.[+257]=2;
39.[+258]=2;
39.[+259]=2;
39.[+261]=2;
39.[+262]=2;
39.[+263]=2;
39.[+264]=2;
39.[+265]=2;
39.[+266]=2;
39.[+267]=2;
39.[+268]=2;
39.[+269]=2;
39.[+270]=2;
39.[+271]=2;
39.[+272]=2;
// MAESTRO ----------------------------
39.[+67]=1;
39.[+56]=1;
39.[+57]=1;
39.[+58]=1;
39.[+59]=1;
// VISA -------------------------------
39.[+4]=5;
// DINERS -----------------------------
39.[+36]=4;
39.[+38]=4;
39.[+39]=4;
// AMEX -------------------------------
39.[+34]=6;
39.[+37]=6;
// JCB --------------------------------
39.[+35]=3;
//
// **********************************************************************************************************************************************
// Terminal type 40 (SK SLOVENSKA SPORITELNA):
// The card recognition is based on Application ID, Card Type and Application Pref. Name.
// **********************************************************************************************************************************************
// MAESTRO ------------------------
40.[+a0000000043060]=1;
40.[+a0000001570021]=1;
40.[+a0000000101030]=1;
40.[+a0000001570022]=1;
40.[+maestro]=1;
// MASTERCARD ---------------------
40.[+a0000000041010]=2;
40.[+a0000001570020]=2;
40.[a0000001574444]=2;
40.[a0000000049999C000020001]=2;
40.[mastercard]=2;
// JCB ---------------------------
40.[+a0000000651010]=3;
40.[+a0000001570040]=3;
40.[jcb]=3;
// DINERS ------------------------
40.[+a0000001884443]=4;
40.[+a0000001574443]=4;
40.[+a0000001570060]=4;
40.[+a0000001523010]=4;
40.[diners]=4;
40.[dci]=4;
// VISA --------------------------
40.[+a0000000031010]=5;
40.[+a0000001570030]=5;
40.[a0000001574458]=5;
40.[a00000000300037562]=5;
40.[visa]=5;
// VISA ELECTRON------------------
40.[+a0000000032010]=10;
40.[+a0000001570031]=10;
40.[visa electron]=10;
// V PAY --------------------------
40.[+a0000000032020]=11;
40.[a0000001574452]=11;
40.[v pay]=11;
40.[vpay]=11;
40.[v-pay]=11;
// AMERICAN EXPRESS --------------
40.[+a0000001570010]=6;
40.[+a00000002501]=6;
40.[+a0000000291010]=6;
40.[+american]=6;
40.[amex]=6;
//
// **********************************************************************************************************************************************
// Terminal type 41 (CH MFGroup Payment Server Gateway):
// The card recognition is based on CardType, Card Number Prefix and/or CardName
// **********************************************************************************************************************************************
// EXLIBRIS GESCHENKKARTE -------------
41.[10]=165,Exlibris Geschenkkarte;
// GROUPCARD (Loyalty Gift Card AG)------
41.[+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 42 (DE/AT/CH/LU easycash PrePayPlus):
// The card recognition is based on Card Number Prefix
// **********************************************************************************************************************************************
// HORNBACH PROFICARD LUX  -----------
42.[+60038803]=420,HORNBACH PROFICARD;
// HORNBACH GUTSCHEINKARTE-------------
42.[+928000116]=422,HORNBACH GUTSCHEINKARTE;
//
// **********************************************************************************************************************************************
// Terminal type 43 (DE/AT/CH Cards & Systems IBRO30):
// The card recognition is based on Card Number Prefix
// **********************************************************************************************************************************************
// HORNBACH ProfiCard AT--------------
43.[+600388021]=420,HORNBACH ProfiCard;
43.[+600388024]=420,HORNBACH ProfiCard;
// HORNBACH ProfiCard CH--------------
43.[+600388061]=420,HORNBACH ProfiCard;
43.[+600388064]=420,HORNBACH ProfiCard;
// HORNBACH Projektweltkarte AT-------
43.[+600388022]=421,HORNBACH ProjektWelt;
//
// *********************************************************************************************************************************************
// Terminal type 44 (CH INNOCARD ICP BIA / ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
44.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
44.[5]=9,GIROCARD;
// EAPS -------------------------------
44.[7]=8,EAPS;
// Lastschrift ------
44.[9]=409,LASTSCHRIFT;
44.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
44.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
44.[13]=11,V PAY;
// BSW --------------------------------
44.[40]=404,BSW;
// POSTCARD ----------------------------
44.[69]=102,POSTFINANCE CARD;
44.[236]=102,POSTFINANCE CARD;
44.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
44.[46]=1;
// MASTERCARD -------------------------
44.[6]=2;

// DINERS -----------------------------
44.[12]=4;
// VISA -------------------------------
44.[10]=5;
// AMEX -------------------------------
44.[8]=6;
// JCB --------------------------------
44.[14]=3;
// HORNBACH Profi ---------------------
44.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
44.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
44.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
44.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
44.[42]=442,Comfort Card;
// Breuninger--------------------------
44.[38]=438,Breuninger;
44.[+352]=452,Paysafecard;
44.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
44.[23]=160,boncard POINTS;
44.[25]=161,boncard PAY;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
44.[218]=175;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
44.[+67]=1;
// COMFORT CARD -----------------------
44.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
44.[+5]=2;
44.[+222]=2;
44.[+223]=2;
44.[+224]=2;
44.[+225]=2;
44.[+226]=2;
44.[+227]=2;
44.[+228]=2;
44.[+229]=2;
44.[+230]=2;
44.[+231]=2;
44.[+232]=2;
44.[+233]=2;
44.[+234]=2;
44.[+235]=2;
44.[+236]=2;
44.[+237]=2;
44.[+238]=2;
44.[+239]=2;
44.[+240]=2;
44.[+241]=2;
44.[+242]=2;
44.[+243]=2;
44.[+244]=2;
44.[+245]=2;
44.[+246]=2;
44.[+247]=2;
44.[+248]=2;
44.[+249]=2;
44.[+250]=2;
44.[+251]=2;
44.[+252]=2;
44.[+253]=2;
44.[+254]=2;
44.[+255]=2;
44.[+256]=2;
44.[+257]=2;
44.[+258]=2;
44.[+259]=2;
44.[+261]=2;
44.[+262]=2;
44.[+263]=2;
44.[+264]=2;
44.[+265]=2;
44.[+266]=2;
44.[+267]=2;
44.[+268]=2;
44.[+269]=2;
44.[+270]=2;
44.[+271]=2;
44.[+272]=2;
// DINERS -----------------------------
44.[+36]=4;
// VISA -------------------------------
44.[+4]=5;
// AMEX -------------------------------
44.[+37]=6;
// JCB --------------------------------
44.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
44.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  -------------------------------
44.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
44.[+a0000001574480]=186;
// SWISS PAY
44.[a0000008010001]=2000,SWISS PAY;
44.[37]=2000,SWISS PAY;
// SWISS PAY 2
44.[a00000015744BD]=2001,SWISS PAY 2;
44.[37]=2001,SWISS PAY 2;
//
// *******************************************************************************************************************************************
// Terminal type 45 CCV NL Vx810 (O.P.I)
// The card recognition is based on the CARDCIRCUIT
// *******************************************************************************************************************************************
// MAESTRO ----------------------------
45.[+maes]=1,Maestro;
// MASTERCARD -------------------------
45.[ecmc]=2,MasterCard;
// JCB --------------------------------
45.[jcb]=3,JCB;
// DINERS -----------------------------
45.[+diner]=4,Diners Club;
// VISA -------------------------------
45.[+visa]=5;
// V PAY ------------------------------
45.[vpay]=11,V PAY;
// AMERICAN EXPRESS -------------------
45.[amex]=6,American Express;
// CHINA UNION PAY --------------------
45.[cup]=12,China Union Pay;
// BANCONTACT/MISTER CASH -------------
45.[bank]=700,Bancontact/Mister Cash;
// GIRO -------------------------------
45.[giro]=701;
// COMORT CARD ------------------------
45.[cmfc]=704,Comfort Card;
// PRLI -------------------------------
45.[prli]=705;
// ELECTRONISCHE POST WISSEL (EPWS) ---
45.[epws]=702,Electronische Post Wissel;
// BASIC CARD -------------------------
45.[baca]=706,Basic Card;
// HORNBACH PROFICARD NL---------------
45.[ccv]=718,Hornbach Proficard;
// Chipknip ----------------------------
45.[ck  ]=719,Chipknip;
45.[chip]=719,Chipknip;
//
// *******************************************************************************************************************************************
// Terminal type 46 BAXI (SE/DK/NO/FI/BALTIC STATES/PL):
// The card recognition is based on ISSUER-ID.
// *******************************************************************************************************************************************
// VISA -------------------------------
46.[3]=5,VISA Udenlandsk;
46.[39]=5,VISA;
46.[57]=1101,VISA Danmark;
// MASTERCARD--------------------------
46.[4]=2,MASTERCARD Udenlandsk;
46.[58]=1102,MASTERCARD DK;
// AMEX -------------------------------
46.[5]=6,AMEX Udenlandsk;
46.[61]=1105,AMEX DK;
// DINERS -----------------------------
46.[6]=4,DINERS Udenlandsk;
46.[60]=1104,DINERS DK;
// JCB - Japanese Credit Banc ---------
46.[11]=3,JCB;
// MAESTRO ----------------------------
46.[14]=1,MAESTRO Udenlandsk;
46.[59]=1103,MAESTRO DK;
// CUP --------------------------------
46.[68]=12,CUP;
// DOMESTIC CARDS ---------------------
46.[50]=1100,DanKort;
46.[64]=1106,PBS Gavekort;
//46.[64]=1106,PBS Kjedekort;
46.[77]=1107,PBS Centerkort;
46.[65]=1108,Forbrugsforeningen;
46.[66]=1109,Ikano Finans;
46.[67]=1110,SparExpress;
46.[30]=1111,bankaxept;
//
// *******************************************************************************************************************************************
// Terminal type 47 Reserved for future use
// *******************************************************************************************************************************************
//
//*******************************************************************************************************************************************
// Terminal type 48 (CH ICP BIA / ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *******************************************************************************************************************************************
// CHINA UNION PAY ---------------------
48.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
48.[5]=9,GIROCARD;
// EAPS -------------------------------
48.[7]=8,EAPS;
// Lastschrift ------
48.[9]=409,LASTSCHRIFT;
48.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
48.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
48.[13]=11,V PAY;
// BSW --------------------------------
48.[40]=404,BSW;
// POSTCARD ----------------------------
48.[69]=102,POSTFINANCE CARD;
48.[236]=102,POSTFINANCE CARD;
48.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
48.[2]=1;
// MASTERCARD -------------------------
48.[6]=2;
// DINERS -----------------------------
48.[12]=4;
// VISA -------------------------------
48.[10]=5;
// AMEX -------------------------------
48.[8]=6;
// JCB --------------------------------
48.[14]=3;
// HORNBACH Profi ---------------------
48.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
48.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
48.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
48.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
48.[42]=442,Comfort Card;
// Breuninger--------------------------
48.[38]=438,Breuninger;
48.[+352]=452,Paysafecard;
48.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
48.[23]=160,boncard POINTS;
48.[25]=161,boncard PAY;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
48.[+67]=1;
// COMFORT CARD -----------------------
48.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
48.[+5]=2;
48.[+222]=2;
48.[+223]=2;
48.[+224]=2;
48.[+225]=2;
48.[+226]=2;
48.[+227]=2;
48.[+228]=2;
48.[+229]=2;
48.[+230]=2;
48.[+231]=2;
48.[+232]=2;
48.[+233]=2;
48.[+234]=2;
48.[+235]=2;
48.[+236]=2;
48.[+237]=2;
48.[+238]=2;
48.[+239]=2;
48.[+240]=2;
48.[+241]=2;
48.[+242]=2;
48.[+243]=2;
48.[+244]=2;
48.[+245]=2;
48.[+246]=2;
48.[+247]=2;
48.[+248]=2;
48.[+249]=2;
48.[+250]=2;
48.[+251]=2;
48.[+252]=2;
48.[+253]=2;
48.[+254]=2;
48.[+255]=2;
48.[+256]=2;
48.[+257]=2;
48.[+258]=2;
48.[+259]=2;
48.[+261]=2;
48.[+262]=2;
48.[+263]=2;
48.[+264]=2;
48.[+265]=2;
48.[+266]=2;
48.[+267]=2;
48.[+268]=2;
48.[+269]=2;
48.[+270]=2;
48.[+271]=2;
48.[+272]=2;
// DINERS -----------------------------
48.[+36]=4;
// VISA -------------------------------
48.[+4]=5;
// AMEX -------------------------------
48.[+37]=6;
// JCB --------------------------------
48.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
48.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  -------------------------------
48.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
48.[+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 49 (US VERIFONE PCCHARGE THINCLIENT):
// The card recognition is based on BIN range (=PAN Prefix)
// **********************************************************************************************************************************************
// DCCB - Diners Club -----------------
49.[+300]=4,DINERS;
49.[+301]=4,DINERS;
49.[+302]=4,DINERS;
49.[+303]=4,DINERS;
49.[+304]=4,DINERS;
49.[+305]=4,DINERS;
49.[+3095]=4,DINERS;
49.[+3095]=4,DINERS;
49.[+3096]=4,DINERS;
49.[+3097]=4,DINERS;
49.[+3098]=4,DINERS;
49.[+3099]=4,DINERS;
49.[+36]=4,DINERS;
49.[+38]=4,DINERS;
49.[+39]=4,DINERS; 
49.[+60110]=4,DINERS;
49.[+60112]=4,DINERS;
49.[+60113]=4,DINERS;
49.[+60114]=4,DINERS;
49.[+601174]=4,DINERS;
49.[+601177]=4,DINERS;
49.[+601178]=4,DINERS;
49.[+601179]=4,DINERS;
49.[+601186]=4,DINERS;
49.[+601187]=4,DINERS;
49.[+601188]=4,DINERS;
49.[+601189]=4,DINERS;
49.[+60119]=4,DINERS;
49.[+644]=4,DINERS;
49.[+645]=4,DINERS;
49.[+646]=4,DINERS;
49.[+647]=4,DINERS;
49.[+648]=4,DINERS;
49.[+649]=4,DINERS;
49.[+65,-6541,-6556]=4,DINERS;  
// DISCOVER ---------------------------
49.[+6011]=600,DISCOVER;
49.[+6541]=600,DISCOVER;
49.[+6556]=600,DISCOVER; 
// MASTERCARD -------------------------
49.[+500235]=2,MASTERCARD;
49.[+500766]=2,MASTERCARD;
49.[+501012]=2,MASTERCARD;
49.[+502029]=2,MASTERCARD;
49.[+503615]=2,MASTERCARD;
49.[+504507]=2,MASTERCARD;
49.[+504834]=2,MASTERCARD;
49.[+504837]=2,MASTERCARD;
49.[+5049]=2,MASTERCARD;
49.[+51]=2,MASTERCARD;
49.[+52]=2,MASTERCARD;
49.[+53]=2,MASTERCARD;
49.[+54,-545250]=2,MASTERCARD;
49.[+55]=2,MASTERCARD;
49.[+222]=2,MASTERCARD;
49.[+223]=2,MASTERCARD;
49.[+224]=2,MASTERCARD;
49.[+225]=2,MASTERCARD;
49.[+226]=2,MASTERCARD;
49.[+227]=2,MASTERCARD;
49.[+228]=2,MASTERCARD;
49.[+229]=2,MASTERCARD;
49.[+230]=2,MASTERCARD;
49.[+231]=2,MASTERCARD;
49.[+232]=2,MASTERCARD;
49.[+233]=2,MASTERCARD;
49.[+234]=2,MASTERCARD;
49.[+235]=2,MASTERCARD;
49.[+236]=2,MASTERCARD;
49.[+237]=2,MASTERCARD;
49.[+238]=2,MASTERCARD;
49.[+239]=2,MASTERCARD;
49.[+240]=2,MASTERCARD;
49.[+241]=2,MASTERCARD;
49.[+242]=2,MASTERCARD;
49.[+243]=2,MASTERCARD;
49.[+244]=2,MASTERCARD;
49.[+245]=2,MASTERCARD;
49.[+246]=2,MASTERCARD;
49.[+247]=2,MASTERCARD;
49.[+248]=2,MASTERCARD;
49.[+249]=2,MASTERCARD;
49.[+250]=2,MASTERCARD;
49.[+251]=2,MASTERCARD;
49.[+252]=2,MASTERCARD;
49.[+253]=2,MASTERCARD;
49.[+254]=2,MASTERCARD;
49.[+255]=2,MASTERCARD;
49.[+256]=2,MASTERCARD;
49.[+257]=2,MASTERCARD;
49.[+258]=2,MASTERCARD;
49.[+259]=2,MASTERCARD;
49.[+261]=2,MASTERCARD;
49.[+262]=2,MASTERCARD;
49.[+263]=2,MASTERCARD;
49.[+264]=2,MASTERCARD;
49.[+265]=2,MASTERCARD;
49.[+266]=2,MASTERCARD;
49.[+267]=2,MASTERCARD;
49.[+268]=2,MASTERCARD;
49.[+269]=2,MASTERCARD;
49.[+270]=2,MASTERCARD;
49.[+271]=2,MASTERCARD;
49.[+272]=2,MASTERCARD;
// CHINA UNION PAY --------------------
49.[+456351]=12,CHINA UNION PAY;
49.[+602969]=12,CHINA UNION PAY;
49.[+603367]=12,CHINA UNION PAY;
49.[+601382 ]=12,CHINA UNION PAY;
49.[+620]=12,CHINA UNION PAY;
49.[+621]=12,CHINA UNION PAY;
49.[+622]=12,CHINA UNION PAY;
49.[+623]=12,CHINA UNION PAY;
49.[+624]=12,CHINA UNION PAY;
49.[+625]=12,CHINA UNION PAY; 
// MAESTRO ----------------------------
49.[+50]=1,MAESTRO;
49.[+54250]=1,MAESTRO;
49.[+56]=1,MAESTRO;
49.[+57]=1,MAESTRO;
49.[+58]=1,MAESTRO;
49.[+59]=1,MAESTRO;
49.[+60]=1,MAESTRO;
49.[+61]=1,MAESTRO;
49.[+62]=1,MAESTRO;
49.[+63]=1,MAESTRO;
49.[+64]=1,MAESTRO;
49.[+66]=1,MAESTRO;
49.[+67]=1,MAESTRO;
49.[+68]=1,MAESTRO;
49.[+69]=1,MAESTRO;
// VISA ELECTRON -----------------------
49.[+400115]=10,VISA ELECTRON;
49.[+400837]=10,VISA ELECTRON;
49.[+400838]=10,VISA ELECTRON;
49.[+400839]=10,VISA ELECTRON;
49.[+412921]=10,VISA ELECTRON;
49.[+412922]=10,VISA ELECTRON;
49.[+412923]=10,VISA ELECTRON;
49.[+417935]=10,VISA ELECTRON;
49.[+419740]=10,VISA ELECTRON;
49.[+419741]=10,VISA ELECTRON;
49.[+419773]=10,VISA ELECTRON;
49.[+419774]=10,VISA ELECTRON;
49.[+419775]=10,VISA ELECTRON;
49.[+424519]=10,VISA ELECTRON;
49.[+424962]=10,VISA ELECTRON;
49.[+424963]=10,VISA ELECTRON;
49.[+437860]=10,VISA ELECTRON;
49.[+444000]=10,VISA ELECTRON;
49.[+459472]=10,VISA ELECTRON;
49.[+484406]=10,VISA ELECTRON;
49.[+484407]=10,VISA ELECTRON;
49.[+484408]=10,VISA ELECTRON;
49.[+484409]=10,VISA ELECTRON;
49.[+484410]=10,VISA ELECTRON;
49.[+484411]=10,VISA ELECTRON;
49.[+484413]=10,VISA ELECTRON;
49.[+484414]=10,VISA ELECTRON;
49.[+484418]=10,VISA ELECTRON;
49.[+484419]=10,VISA ELECTRON;
49.[+484420]=10,VISA ELECTRON;
49.[+484421]=10,VISA ELECTRON;
49.[+484422]=10,VISA ELECTRON;
49.[+484423]=10,VISA ELECTRON;
49.[+484424]=10,VISA ELECTRON;
49.[+484425]=10,VISA ELECTRON;
49.[+484426]=10,VISA ELECTRON;
49.[+484428]=10,VISA ELECTRON;
49.[+484429]=10,VISA ELECTRON;
49.[+48443]=10,VISA ELECTRON;
49.[+48444]=10,VISA ELECTRON;
49.[+484450]=10,VISA ELECTRON;
49.[+484451]=10,VISA ELECTRON;
49.[+484452]=10,VISA ELECTRON;
49.[+484453]=10,VISA ELECTRON;
49.[+484454]=10,VISA ELECTRON;
49.[+484455]=10,VISA ELECTRON;
49.[+49173]=10,VISA ELECTRON;
49.[+49174]=10,VISA ELECTRON;
49.[+49175]=10,VISA ELECTRON;
// VISA -------------------------------
49.[+4]=5,VISA;
// AMEX -------------------------------
49.[+34]=6,AMERICAN EXPRESS;
49.[+37]=6,AMERICAN EXPRESS;
// JCB --------------------------------
49.[+35]=3,JCB; 
// CBLN - Carte Blanche ---------------
49.[+389]=601;
// JAL- Japan Airlines ----------------
49.[+1800]=7;
49.[+2131]=7;
// EN ROUTE ---------------------------
49.[+201400]=602,EN ROUTE;
//
// **********************************************************************************************************************************************
// Terminal type 50 (AT HOBEX):
// The card recognition is based on the Application IDs and brand names from terminal (e.g. MAESTRO)
// **********************************************************************************************************************************************
// MAESTRO/BANKOMAT (MAESTRO AT) ----------
50.[+maestro]=1;
50.[+a0000000043060]=1;
50.[+a0000000101030]=1;
50.[+a0000001570022]=1;
50.[a0000000031030]=1;
50.[+a0000001570021]=1;
50.[+elv]=201;
// MASTERCARD ---------------------
50.[+mc]=2;
50.[+master]=2;
50.[+a0000000041010]=2;
50.[+a0000001570020]=2;
50.[a0000000000008]=2;
// JCB ---------------------------
50.[+japan]=3;
50.[+jcb]=3;
50.[+a0000000651010]=3;
50.[+a0000001570040]=3;
// DINERS ------------------------
50.[+diners]=4;
50.[+dci]=4;
50.[+a0000001884443]=4;
50.[+a0000001574443]=4;
50.[+a0000001570060]=4;
50.[+a0000001523010]=4;
// VISA --------------------------
50.[+visa]=5;
50.[+a0000000031010]=5;
50.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
50.[+visa electron]=10;
50.[+a0000000032010]=10;
50.[+a0000001570031]=10;
// V PAY -------------------------------
50.[+vpay]=11;
50.[+v pay]=11;
50.[+v-pay]=11;
50.[+a0000000032020]=11;
50.[a0000001574452]=11;
// AMERICAN EXPRESS --------------
50.[+ame]=6;
50.[+a0000001570010]=6;
50.[+a00000002501]=6;
50.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
50.[+china]=12;
50.[+cup]=12;
50.[a000000157446003]=12;
50.[a000000157446004]=12;
50.[a000000157446005]=12;
50.[a000000157446006]=12;
50.[a000000157446009]=12;
50.[a0000001574460]=12;
50.[a000000333010101]=12;
50.[a000000333010102]=12;
//
// **********************************************************************************************************************************************
// Terminal type 51 (CZ WNSP Ingenico ):
// The card recognition is based on Issuer Name
// **********************************************************************************************************************************************
// HORNBACH ---------------------------
51.[+hornbach]=420;
// AMEX -------------------------------
51.[+amex]=6;
51.[+american]=6;
// DCCB - Diners Club -----------------
51.[+diners]=4;
51.[+dci]=4;
// MASTERCARD -------------------------     
51.[+ec/mc]=2;
51.[+master]=2;
// MAESTRO ----------------------------
51.[+maestro]=1;
// VISA -------------------------------
51.[+visa]=5;
// JCB --------------------------------
51.[+jcb]=3;
51.[+japan]=3;
//
// **********************************************************************************************************************************************
// Terminal type 52 (CH Aduno Primus):
// The card recognition is based on Card Name
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
52.[+a0000000043060]=1;
52.[+a0000001570021]=1;
// MASTERCARD -------------------------
52.[+a0000000041010]=2;
52.[+a0000001570020]=2;
// M-Budget
52.[a0000000000008]=2;
// JCB -------------------------------
52.[+a0000000651010]=3;
52.[+a0000001570040]=3;
// DINERS -----------------------------
52.[+a0000001884443]=4;
52.[+a0000001574443]=4;
52.[+a0000001570060]=4;
52.[+a0000001523010]=4;
// VISA -------------------------------
52.[+a0000000031010]=5;
52.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
52.[+a0000000032010]=10;
52.[+a0000001570031]=10;
// V PAY -------------------------------
52.[a0000000032020]=11,V PAY;
52.[a0000001574452]=11,V PAY;
// AMEX -------------------------------
52.[+a00000002501]=6;
52.[+a0000001570010]=6;
52.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
52.[a000000157446003]=12,CHINA UNION PAY;
52.[a000000157446004]=12,CHINA UNION PAY;
52.[a000000157446005]=12,CHINA UNION PAY;
52.[a000000157446006]=12,CHINA UNION PAY;
52.[a000000157446009]=12,CHINA UNION PAY;
52.[a0000001574460]=12,CHINA UNION PAY;
52.[a000000333010101]=12,CHINA UNION PAY;
52.[a000000333010102]=12,CHINA UNION PAY;
52.[a000000333010103]=12,CHINA UNION PAY;
52.[+unionpay]=12,CHINA UNION PAY;
// MAESTRO CH -------------------------
52.[+a0000000101030]=100;
52.[+a0000001570022]=100;
52.[a0000000031030]=100;
// M-CARD -----------------------------
52.[a0000001570100]=101;
52.[a0000001574450]=101;
52.[a000000162010201]=101;
// POSTFINANCE CARD -------------------
52.[a0000001110101]=102;
52.[a0000001999999]=102;
52.[a0000001999998]=102;
52.[a0000001570051]=102;
52.[a0000001570050]=102;
52.[a0000001574445]=102;
52.[a0000001574446]=102;
52.[a0000001110505]=102,POSTCARD;
52.[a0000001570110]=143,PostFinance Mobile Coup.;
52.[a0000001570111]=144,PostFinance Mobile Paym.;
// BERN-CARD --------------------------
52.[+a0000001570107]=103;
// ZUERI-CARD -------------------------
52.[+a0000001570106]=104;
// BASLER-CARD ------------------------
52.[+a0000001570108]=105;
// CASH -------------------------------
52.[+a0000001570023]=106;
// PHARMACARD --------------------------
52.[+a0000001570105]=107;
// MOEVENPICK -------------------------
52.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --------
52.[+a0000001570103]=110;
52.[+a000000157010B]=110;
52.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
52.[a0000001574462]=137;
// MANOR CARD (MYONE) -----------------
52.[+a0000001570104]=111;
52.[+a0000001570109]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
52.[a0000001574444]=116;
52.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
52.[a0000001574458]=134;
52.[a00000000300037562]=134;
52.[a0000000037562]=134;
// POWERCARD (MFGroup)-----------------
52.[+a000000157010D]=117;
52.[a0000001574476]=117;
52.[a0000001574478]=117;
52.[a0000001574479]=117;
// WIR --------------------------------
52.[+a000000157010C]=118;
// REKA  ------------------------------
52.[d7560000011111]=120;
52.[+a000000157010E]=120;
52.[+d7560000010101]=120;
// REKA RAIL --------------------------
52.[d7560000011112]=180;
52.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
52.[d7560000011113]=178;
52.[d7560000010102]=178;
52.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
52.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
52.[+a000000157444E]=122,J-Geschenkkarte;
// ADUNO CARDS --------------
52.[+a0000001574448]=123,Aduno Kundenkarte;
52.[+a0000001574449]=124,Swiss Bonus Card;
52.[+a000000157444A]=125,ABO Card;
52.[a000000157010F]=142,ADUNO TEST;
// MEDIAMARKT --------------
52.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
52.[D7560000300101]=132,M-Budget;
// GIFT CARD (SIX)-----------------------
52.[a0000001574451]=133,Gift Card;
// SWISSCADEAU --------------------------
52.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
52.[a000000157445E]=140,BonGenie/Grieder Card; 
// GIFT CARD (BILLAG) -------------------
52.[a000000157010A]=141,GIFT CARD BILLAG;
// ACCARDA STAEDTEKARTE -----------------
52.[a0000001570112]=115,ACCARDA STAEDTEKARTE;
// KUNDENKARTE OFAC ---------------------
52.[a0000001570113]=145,Kundenkarte OFAC;
// REINVEST SELECT AG COMPANY CARD -----
52.[a0000001574447]=146,REINVEST SELECT;
// KUNDENKARTE PKZ -----------------------
52.[a000000157445D]=147,KUNDENKARTE PKZ; 
// JUBIN CARD ----------------------------
52.[a000000157445F]=148,JUBIN CARD;
// INNOCARD -------------------------------
52.[a0000001574448]=149,INNOCARD KUNDENKARTE;
52.[a000000157444A]=150,ABO CARD;
52.[a0000001574459]=151,SWISS HAPPY CARD;
52.[a0000001574461]=153,DATA STANDARDS;
52.[a0000001574463]=154,SWISSBONUS+;
// GE MONEY BANK ---------------------------
52.[a000000157444B]=156,GE MONEY BANK;
52.[a000000157444F]=157,BARRACUDA TEMPCARD;
// MERGER CARD -----------------------------
52.[a000000157444C]=158,MERGER CARD;
52.[a0000004840001]=158,MERGER CARD;
52.[a000000157444D]=159,MERGER CARD 2;
// BANCOMAT LUXEMBOURG -------------------
52.[a0000001574453]=1000,BANCOMAT LUXEMBOURG;
// AURORA LUXEMBOURG ---------------------
52.[a0000001574454]=1001,AURORA LUXEMBOURG;
// BONCARD -------------------------------
52.[a0000001574455]=160,BONCARD POINTS;
52.[a000000157445B]=161,BONCARD PAY;
// FNAC ----------------------------------
52.[a0000001574456]=162,FNAC-KARTE;
// COMMIT CARD COMPANY CARD --------------
52.[a0000001574457]=163,COMMIT CARD;
// KUNDENKARTE LOEB ----------------------
52.[a000000157445C]=164,KUNDENKARTE LOEB;
// LUNCH-CHECK ---------------------------
52.[a000000157447d]=172;
52.[d7560001150001]=172;
// CCV PAY -------------------------------
52.[a000000157447b]=173;
// COOP EKZ ------------------------------
52.[37]=174;
52.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
52.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
52.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
52.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
52.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
52.[+a0000001574480]=186;
// TWINT -------------------------------
52.[a000000157449e]=184,TWINT;
// SWISS PAY
52.[a0000008010001]=2000,SWISS PAY;
52.[37]=2000,SWISS PAY;
// SWISS PAY 2
52.[a00000015744BD]=2001,SWISS PAY 2;
52.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 53 (CH INGENICO CRP 3.01): 
// The card recognition is based on the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
53.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
53.[+a0000001570021]=1;
// MASTERCARD ---------------------
53.[+a0000000041010]=2;
53.[+a0000001570020]=2;
// M-Budget -----------------------
53.[a0000000000008]=2;
// JCB ---------------------------
53.[+a0000000651010]=3;
53.[+a0000001570040]=3;
// DINERS ------------------------
53.[+a0000001884443]=4;
53.[+a0000001574443]=4;
53.[+a0000001570060]=4;
53.[+a0000001523010]=4;
// VISA --------------------------
53.[+a0000000031010]=5;
53.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
53.[+a0000000032010]=10;
53.[+a0000001570031]=10;
// V PAY -------------------------------
53.[+a0000000032020]=11,V PAY;
53.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
53.[+a0000001570010]=6;
53.[+a00000002501]=6;
53.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
53.[a000000157446003]=12,CHINA UNION PAY;
53.[a000000157446004]=12,CHINA UNION PAY;
53.[a000000157446005]=12,CHINA UNION PAY;
53.[a000000157446006]=12,CHINA UNION PAY;
53.[a000000157446009]=12,CHINA UNION PAY;
53.[a0000001574460]=12,CHINA UNION PAY;
53.[a000000333010101]=12,CHINA UNION PAY;
53.[a000000333010102]=12,CHINA UNION PAY;
53.[a000000333010103]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
53.[+a0000000101030]=100;
53.[+a0000001570022]=100;
53.[a0000000031030]=100;
// M-CARD ------------------------
53.[a0000001570100]=101;
53.[a0000001574450]=101;
// POSTCARD
53.[+a0000001110101]=102;
53.[+a0000001999999]=102;
53.[+a0000001999998]=102;
53.[+a0000001570051]=102;
53.[+a0000001570050]=102;
53.[+a0000001574445]=102;
53.[+a0000001574446]=102;
53.[a0000001110505]=102,POSTCARD;
// BERN CARD --------------------
53.[+a0000001570107]=103;
// ZUERI CARD -------------------
53.[+a0000001570106]=104;
// BASLER CARD ------------------
53.[+a0000001570108]=105;
// CASH -------------------------
53.[+a0000001570023]=106;
// PHARMACARD --------------------
53.[+a0000001570105]=107;
// MOEVENPICK -------------------
53.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
53.[+a0000001570103]=110;
53.[+a000000157010B]=110;
53.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
53.[a0000001574462]=137;
// MY ONE -----------------------
53.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
53.[a0000001574444]=116;
53.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
53.[a0000001574458]=134;
53.[a00000000300037562]=134;
53.[a0000000037562]=134;
// POWER CARD -------------------
53.[+a000000157010D]=117;
53.[a0000001574476]=117;
53.[a0000001574478]=117;
53.[a0000001574479]=117;
// WIR --------------------------
53.[+a000000157010C]=118;
// REKA  ------------------------------
53.[d7560000011111]=120;
53.[+a000000157010E]=120;
53.[+D7560000010101]=120;
// REKA RAIL --------------------------
53.[d7560000011112]=180;
53.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
53.[d7560000011113]=178;
53.[d7560000010102]=178;
53.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
53.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
53.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
53.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
53.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
53.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
53.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
53.[a0000001574455]=160,BONARD POINTS;
53.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK
53.[a000000157447d]=172;
53.[d7560001150001]=172;
// CCV PAY
53.[a000000157447b]=173;
// COOP EKZ
53.[37]=174;
53.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
53.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
53.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
53.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
53.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
53.[+a0000001574480]=186;
// SWISS PAY
53.[a0000008010001]=2000,SWISS PAY;
53.[37]=2000,SWISS PAY;
// SWISS PAY 2
53.[a00000015744BD]=2001,SWISS PAY 2;
53.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 54 (FI Micros Point):
// The card recognition is based 1. on the AID Index and 2. on the settlement ID
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
54.[04]=1;
// MASTERCARD -------------------------
54.[03]=2;
// VISA -------------------------------
54.[01]=5;
// VISA ELECTRON ----------------------
54.[02]=10;
// AMEX -------------------------------
54.[08]=6;
// DINERS -----------------------------
54.[09]=4;
// JCB --------------------------------
54.[10]=3;
54.[M5]=3;
// PANKKIKORTTI (BANK CARD) -----------
54.[05]=1112;
// MASTERCARD PANKKIKORTTI YHDISTELMÄKORTTI
54.[12]=1113; 
// VISA PANKKIKORTTI YHDISTELMÄKORTTI--
54.[13]=1114;
// OPTION -----------------------------
54.[14]=1115;
//
// **********************************************************************************************************************************************
// Terminal type 55 (DE Verifone H5000 ZVT Rev. 1305):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *******************************************************************************************************************************************
// CHINA UNION PAY ---------------------
55.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
55.[5]=9,GIROCARD;
// EAPS -------------------------------
55.[7]=8,EAPS;
// Lastschrift ------
55.[9]=409,LASTSCHRIFT;
55.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
55.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
55.[13]=11,V PAY;
// BSW --------------------------------
55.[40]=404,BSW;
// MAESTRO ----------------------------
55.[46]=1;
// MASTERCARD -------------------------
55.[6]=2;
// DINERS -----------------------------
55.[12]=4;
// VISA -------------------------------
55.[10]=5;
// AMEX -------------------------------
55.[8]=6;
// JCB --------------------------------
55.[14]=3;
// MAESTRO DE / ELV -------------------
55.[2ELV]=410;
// MAESTRO DE / OLV -------------------
55.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
55.[2CASH]=412;
// MAESTRO DE / default ---------------
55.[2]=401;
// GELDKARTE --------------------------
55.[30]=402;
// HORNBACH Profi ---------------------
55.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
55.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
55.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
55.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
55.[42]=442,Comfort Card;
// Breuninger -------------------------
55.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
55.[+414]=414,AnniCard;
55.[+AnniCard]=414,AnniCard;
//
55.[212]=441,T-D1 XTRA;
55.[80]=441,T-D1 XTRA;
55.[+T-D1 XTRA]=441,T-D1 XTRA;
55.[+T-D1]=441,T-D1 XTRA;
55.[+XTRA]=441,T-D1 XTRA;
//
55.[191]=479,Vodafone CallYa;
55.[+Vodafone CallYa]=479,Vodafone CallYa;
55.[+Vodafone]=479,Vodafone CallYa;
55.[+CallYa]=479,Vodafone CallYa;
//
55.[190]=480,E-Plus Free&Easy;
55.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
55.[+E-Plus]=480,E-Plus Free&Easy;
55.[+Free&Easy]=480,E-Plus Free&Easy;
//
55.[188]=481,O2 LOOP;
55.[+O2 LOOP]=481,O2 LOOP;
55.[+O2]=481,O2 LOOP;
55.[+LOOP]=481,O2 LOOP;
//
55.[+431]=431,Premiere Flex;
55.[+Premiere Flex]=431,Premiere Flex;
55.[+Premiere]=431,Premiere Flex;
55.[+Flex]=431,Premiere Flex;
//
55.[189]=405,gobananas;
55.[+gobananas]=405,gobananas;
//
55.[180]=452,Paysafecard Blue;
55.[181]=452,Paysafecard Red;
55.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
55.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
55.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
55.[807]=484,SCHLECKER PREPAID;
55.[249]=486,smobil;
55.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
55.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
55.[246]=488,FONIC;
55.[206]=489,BILDmobil;
55.[205]=490,Blau.de;
55.[207]=491,Congstar;
55.[475]=492,Lebara Mobile;
55.[248]=493,Simyo;
55.[884]=494,Lycamobile;
55.[917]=495,GT mobile;
// AVS -----------------------------
55.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
55.[+6010822]=497,Alnatura Gutscheinkarte;
// 220 TLAPA-KARTE---------------------
55.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
55.[+67]=1;
// COMFORT CARD -----------------------
55.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
55.[+5]=2;
55.[+222]=2;
55.[+223]=2;
55.[+224]=2;
55.[+225]=2;
55.[+226]=2;
55.[+227]=2;
55.[+228]=2;
55.[+229]=2;
55.[+230]=2;
55.[+231]=2;
55.[+232]=2;
55.[+233]=2;
55.[+234]=2;
55.[+235]=2;
55.[+236]=2;
55.[+237]=2;
55.[+238]=2;
55.[+239]=2;
55.[+240]=2;
55.[+241]=2;
55.[+242]=2;
55.[+243]=2;
55.[+244]=2;
55.[+245]=2;
55.[+246]=2;
55.[+247]=2;
55.[+248]=2;
55.[+249]=2;
55.[+250]=2;
55.[+251]=2;
55.[+252]=2;
55.[+253]=2;
55.[+254]=2;
55.[+255]=2;
55.[+256]=2;
55.[+257]=2;
55.[+258]=2;
55.[+259]=2;
55.[+261]=2;
55.[+262]=2;
55.[+263]=2;
55.[+264]=2;
55.[+265]=2;
55.[+266]=2;
55.[+267]=2;
55.[+268]=2;
55.[+269]=2;
55.[+270]=2;
55.[+271]=2;
55.[+272]=2;
// DINERS -----------------------------
55.[+36]=4;
// VISA -------------------------------
55.[+4]=5;
// AMEX -------------------------------
55.[+37]=6;
// JCB --------------------------------
55.[+35]=3;
// Alipay ---------------------
55.[97]=17,AliPay;
// Bluecode --------------------------------
55.[260]=22,BLUECODE;
//
// **********************************************************************************************************************************************
// Terminal type 56 (DE Verifone VX 820 ZVT Rev. 1305):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
56.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
56.[5]=9,GIROCARD;
// EAPS -------------------------------
56.[7]=8,EAPS;
// Lastschrift ------------------------
56.[9]=409,LASTSCHRIFT;
56.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
56.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
56.[13]=11,V PAY;
// BSW --------------------------------
56.[40]=404,BSW;
// MAESTRO ----------------------------
56.[46]=1;
// MASTERCARD -------------------------
56.[6]=2;
// DINERS -----------------------------
56.[12]=4;
// VISA -------------------------------
56.[10]=5;
// AMEX -------------------------------
56.[8]=6;
// JCB --------------------------------
56.[14]=3;
// MAESTRO DE / ELV -------------------
56.[2ELV]=410;
// MAESTRO DE / OLV -------------------
56.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
56.[2CASH]=412;
// MAESTRO DE / default ---------------
56.[2]=401;
// GELDKARTE --------------------------
56.[30]=402;
// IKEA FAMILY PLUS -------------------
56.[76]=403;
// HORNBACH Profi ---------------------
56.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
56.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
56.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
56.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
56.[42]=442,Comfort Card;
// Breuninger-- -----------------------
56.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
56.[+414]=414,AnniCard;
56.[+AnniCard]=414,AnniCard;
//
56.[212]=441,T-D1 XTRA;
56.[+T-D1 XTRA]=441,T-D1 XTRA;
56.[+T-D1]=441,T-D1 XTRA;
56.[+XTRA]=441,T-D1 XTRA;
//
56.[191]=479,Vodafone CallYa;
56.[+Vodafone CallYa]=479,Vodafone CallYa;
56.[+Vodafone]=479,Vodafone CallYa;
56.[+CallYa]=479,Vodafone CallYa;
//
56.[190]=480,E-Plus Free&Easy;
56.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
56.[+E-Plus]=480,E-Plus Free&Easy;
56.[+Free&Easy]=480,E-Plus Free&Easy;
//
56.[188]=481,O2 LOOP;
56.[+O2 LOOP]=481,O2 LOOP;
56.[+O2]=481,O2 LOOP;
56.[+LOOP]=481,O2 LOOP;
//
56.[+431]=431,Premiere Flex;
56.[+Premiere Flex]=431,Premiere Flex;
56.[+Premiere]=431,Premiere Flex;
56.[+Flex]=431,Premiere Flex;
//
56.[189]=405,gobananas;
56.[+gobananas]=405,gobananas;
//
56.[+352]=452,Paysafecard;
56.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
56.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
56.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
56.[807]=484,SCHLECKER PREPAID;
56.[249]=486,smobil;
56.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
56.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
56.[246]=488,FONIC;
56.[206]=489,BILDmobil;
56.[205]=490,Blau.de;
56.[207]=491,Congstar;
56.[475]=492,Lebara Mobile;
56.[248]=493,Simyo;
56.[884]=494,Lycamobile;
56.[917]=495,GT mobile;
// AVS -----------------------------
56.[253]=496,AVS Gutscheinkarte;
//
// 220 TLAPA-KARTE---------------------
56.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
56.[+67]=1;
// COMFORT CARD -----------------------
56.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
56.[+5]=2;
56.[+222]=2;
56.[+223]=2;
56.[+224]=2;
56.[+225]=2;
56.[+226]=2;
56.[+227]=2;
56.[+228]=2;
56.[+229]=2;
56.[+230]=2;
56.[+231]=2;
56.[+232]=2;
56.[+233]=2;
56.[+234]=2;
56.[+235]=2;
56.[+236]=2;
56.[+237]=2;
56.[+238]=2;
56.[+239]=2;
56.[+240]=2;
56.[+241]=2;
56.[+242]=2;
56.[+243]=2;
56.[+244]=2;
56.[+245]=2;
56.[+246]=2;
56.[+247]=2;
56.[+248]=2;
56.[+249]=2;
56.[+250]=2;
56.[+251]=2;
56.[+252]=2;
56.[+253]=2;
56.[+254]=2;
56.[+255]=2;
56.[+256]=2;
56.[+257]=2;
56.[+258]=2;
56.[+259]=2;
56.[+261]=2;
56.[+262]=2;
56.[+263]=2;
56.[+264]=2;
56.[+265]=2;
56.[+266]=2;
56.[+267]=2;
56.[+268]=2;
56.[+269]=2;
56.[+270]=2;
56.[+271]=2;
56.[+272]=2;
// DINERS -----------------------------
56.[+36]=4;
// VISA -------------------------------
56.[+4]=5;
// AMEX -------------------------------
56.[+37]=6;
// JCB --------------------------------
56.[+35]=3;
// AirPlus Company --------------------------------
56.[+AirPlus]=5000,AirPlus;
56.[+992003]=5000,AirPlus;
// Alipay ---------------------
56.[97]=17,AliPay;
// Bluecode --------------------------------
56.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 57 (DE Ingenico IPP480 ZVT/A32.de)
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
57.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
57.[5]=9,GIROCARD;
// EAPS -------------------------------
57.[7]=8,EAPS;
// Lastschrift ------
57.[9]=409,LASTSCHRIFT;
57.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
57.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
57.[13]=11,V PAY;
// BSW --------------------------------
57.[40]=404,BSW;
// AUSTRIA CARD -----------------------
57.[217]=217,Austria Card;
// MAESTRO ----------------------------
57.[46]=1;
// MASTERCARD -------------------------
57.[6]=2;
// DINERS -----------------------------
57.[12]=4;
// VISA -------------------------------
57.[10]=5;
// AMEX -------------------------------
57.[8]=6;
// JCB --------------------------------
57.[14]=3;
// MAESTRO DE / ELV -------------------
57.[2ELV]=410;
// MAESTRO DE / OLV -------------------
57.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
57.[2CASH]=412;
// MAESTRO DE / default ---------------
57.[2]=401;
// GELDKARTE --------------------------
57.[30]=402;
// IKEA FAMILY PLUS -------------------
57.[76]=403;
// HORNBACH Profi ---------------------
57.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
57.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
57.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
57.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
57.[42]=442,Comfort Card;
// Breuninger--------------------------
57.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
57.[+414]=414,AnniCard;
57.[+AnniCard]=414,AnniCard;
//
57.[212]=441,T-D1 XTRA;
57.[+T-D1 XTRA]=441,T-D1 XTRA;
57.[+T-D1]=441,T-D1 XTRA;
57.[+XTRA]=441,T-D1 XTRA;
//
57.[191]=479,Vodafone CallYa;
57.[+Vodafone CallYa]=479,Vodafone CallYa;
57.[+Vodafone]=479,Vodafone CallYa;
57.[+CallYa]=479,Vodafone CallYa;
//
57.[+80]=480,E-Plus Free&Easy;
57.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
57.[+E-Plus]=480,E-Plus Free&Easy;
57.[+Free&Easy]=480,E-Plus Free&Easy;
//
57.[188]=481,O2 LOOP;
57.[+O2 LOOP]=481,O2 LOOP;
57.[+O2]=481,O2 LOOP;
57.[+LOOP]=481,O2 LOOP;
//
57.[+431]=431,Premiere Flex;
57.[+Premiere Flex]=431,Premiere Flex;
57.[+Premiere]=431,Premiere Flex;
57.[+Flex]=431,Premiere Flex;
//
57.[189]=405,gobananas;
57.[+gobananas]=405,gobananas;
//
57.[+352]=452,Paysafecard;
57.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
57.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
57.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
57.[807]=484,SCHLECKER PREPAID;
57.[249]=486,smobil;
57.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
57.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
57.[246]=488,FONIC;
57.[206]=489,BILDmobil;
57.[205]=490,Blau.de;
57.[207]=491,Congstar;
57.[475]=492,Lebara Mobile;
57.[248]=493,Simyo;
57.[884]=494,Lycamobile;
57.[917]=495,GT mobile;
// AVS -----------------------------
57.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
57.[+6010822]=497,Alnatura Gutscheinkarte;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
57.[+67]=1;
// COMFORT CARD -----------------------
57.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
57.[+5]=2;
57.[+222]=2;
57.[+223]=2;
57.[+224]=2;
57.[+225]=2;
57.[+226]=2;
57.[+227]=2;
57.[+228]=2;
57.[+229]=2;
57.[+230]=2;
57.[+231]=2;
57.[+232]=2;
57.[+233]=2;
57.[+234]=2;
57.[+235]=2;
57.[+236]=2;
57.[+237]=2;
57.[+238]=2;
57.[+239]=2;
57.[+240]=2;
57.[+241]=2;
57.[+242]=2;
57.[+243]=2;
57.[+244]=2;
57.[+245]=2;
57.[+246]=2;
57.[+247]=2;
57.[+248]=2;
57.[+249]=2;
57.[+250]=2;
57.[+251]=2;
57.[+252]=2;
57.[+253]=2;
57.[+254]=2;
57.[+255]=2;
57.[+256]=2;
57.[+257]=2;
57.[+258]=2;
57.[+259]=2;
57.[+261]=2;
57.[+262]=2;
57.[+263]=2;
57.[+264]=2;
57.[+265]=2;
57.[+266]=2;
57.[+267]=2;
57.[+268]=2;
57.[+269]=2;
57.[+270]=2;
57.[+271]=2;
57.[+272]=2;
// DINERS -----------------------------
57.[+36]=4;
// VISA -------------------------------
57.[+4]=5;
// AMEX -------------------------------
57.[+37]=6;
// JCB --------------------------------
57.[+35]=3;
// Alipay ---------------------
57.[97]=17,AliPay;
// Bluecode --------------------------------
57.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 57 (DE Ingenico IPP480 ZVT/A32.de) >>>>>> Atos Card-IDs <<<<<<
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
//57.[21]=1;
// MASTERCARD -------------------------
//57.[1]=2;
// DINERS -----------------------------
//57.[4]=4;
// VISA -------------------------------
//57.[3]=5;
// AMEX -------------------------------
//57.[2]=6;
// MAESTRO DE / ELV -------------------
//57.[0ELV]=410;
// MAESTRO DE / OLV -------------------
//57.[0OLV]=411;
// MAESTRO DE / ec-CASH ---------------
//57.[0CASH]=412;
// MAESTRO DE / default ---------------
//57.[0]=401;
// GELDKARTE --------------------------
//57.[30]=402;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
//57.[+67]=1;
// COMFORT CARD -----------------------
//57.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
//57.[+5]=2;
// DINERS -----------------------------
//57.[+36]=4;
// VISA -------------------------------
//57.[+4]=5;
// AMEX -------------------------------
//57.[+37]=6;
// JCB --------------------------------
//57.[+35]=3;
//
// **********************************************************************************************************************************************
// Terminal type 58 (DE Ingenico ICT250 ZVT/A32.de)
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
58.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
58.[5]=9,GIROCARD;
// EAPS -------------------------------
58.[7]=8,EAPS;
// Lastschrift ------
58.[9]=409,LASTSCHRIFT;
58.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
58.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
58.[13]=11,V PAY;
// BSW --------------------------------
58.[40]=404,BSW;
// AUSTRIA CARD -----------------------
58.[217]=217,Austria Card;
// MAESTRO ----------------------------
58.[46]=1;
// MASTERCARD -------------------------
58.[6]=2;
// DINERS -----------------------------
58.[12]=4;
// VISA -------------------------------
58.[10]=5;
// AMEX -------------------------------
58.[8]=6;
// JCB --------------------------------
58.[14]=3;
// MAESTRO DE / ELV -------------------
58.[2ELV]=410;
// MAESTRO DE / OLV -------------------
58.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
58.[2CASH]=412;
// MAESTRO DE / default ---------------
58.[2]=401;
// GELDKARTE --------------------------
58.[30]=402;
// IKEA FAMILY PLUS -------------------
58.[76]=403;
// HORNBACH Profi ---------------------
58.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
58.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
58.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
58.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
58.[42]=442,Comfort Card;
// Breuninger--------------------------
58.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
58.[+414]=414,AnniCard;
58.[+AnniCard]=414,AnniCard;
//
58.[212]=441,T-D1 XTRA;
58.[+T-D1 XTRA]=441,T-D1 XTRA;
58.[+T-D1]=441,T-D1 XTRA;
58.[+XTRA]=441,T-D1 XTRA;
//
58.[191]=479,Vodafone CallYa;
58.[+Vodafone CallYa]=479,Vodafone CallYa;
58.[+Vodafone]=479,Vodafone CallYa;
58.[+CallYa]=479,Vodafone CallYa;
//
58.[+80]=480,E-Plus Free&Easy;
58.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
58.[+E-Plus]=480,E-Plus Free&Easy;
58.[+Free&Easy]=480,E-Plus Free&Easy;

//
58.[188]=481,O2 LOOP;
58.[+O2 LOOP]=481,O2 LOOP;
58.[+O2]=481,O2 LOOP;
58.[+LOOP]=481,O2 LOOP;
//
58.[+431]=431,Premiere Flex;
58.[+Premiere Flex]=431,Premiere Flex;
58.[+Premiere]=431,Premiere Flex;
58.[+Flex]=431,Premiere Flex;
//
58.[189]=405,gobananas;
58.[+gobananas]=405,gobananas;
//
58.[+352]=452,Paysafecard;
58.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
58.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
58.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
58.[807]=484,SCHLECKER PREPAID;
58.[249]=486,smobil;
58.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
58.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
58.[246]=488,FONIC;
58.[206]=489,BILDmobil;
58.[205]=490,Blau.de;
58.[207]=491,Congstar;
58.[475]=492,Lebara Mobile;
58.[248]=493,Simyo;
58.[884]=494,Lycamobile;
58.[917]=495,GT mobile;
// AVS -----------------------------
58.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
58.[+6010822]=497,Alnatura Gutscheinkarte;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
58.[+67]=1;
// COMFORT CARD -----------------------
58.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
58.[+5]=2;
58.[+222]=2;
58.[+223]=2;
58.[+224]=2;
58.[+225]=2;
58.[+226]=2;
58.[+227]=2;
58.[+228]=2;
58.[+229]=2;
58.[+230]=2;
58.[+231]=2;
58.[+232]=2;
58.[+233]=2;
58.[+234]=2;
58.[+235]=2;
58.[+236]=2;
58.[+237]=2;
58.[+238]=2;
58.[+239]=2;
58.[+240]=2;
58.[+241]=2;
58.[+242]=2;
58.[+243]=2;
58.[+244]=2;
58.[+245]=2;
58.[+246]=2;
58.[+247]=2;
58.[+248]=2;
58.[+249]=2;
58.[+250]=2;
58.[+251]=2;
58.[+252]=2;
58.[+253]=2;
58.[+254]=2;
58.[+255]=2;
58.[+256]=2;
58.[+257]=2;
58.[+258]=2;
58.[+259]=2;
58.[+261]=2;
58.[+262]=2;
58.[+263]=2;
58.[+264]=2;
58.[+265]=2;
58.[+266]=2;
58.[+267]=2;
58.[+268]=2;
58.[+269]=2;
58.[+270]=2;
58.[+271]=2;
58.[+272]=2;
// DINERS -----------------------------
58.[+36]=4;
// VISA -------------------------------
58.[+4]=5;
// AMEX -------------------------------
58.[+37]=6;
// JCB --------------------------------
58.[+35]=3;
// Alipay ---------------------
58.[97]=17,AliPay;
// Bluecode --------------------------------
58.[260]=22,BLUECODE;
//
// **********************************************************************************************************************************************
// Terminal type 58 (DE Ingenico ICT250 ZVT/A32.de) >>>>>> Atos Card-IDs <<<<<<
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
//58.[21]=1;
// MASTERCARD -------------------------
//58.[1]=2;
// DINERS -----------------------------
//58.[4]=4;
// VISA -------------------------------
//58.[3]=5;
// AMEX -------------------------------
//58.[2]=6;
// MAESTRO DE / ELV -------------------
//58.[0ELV]=410;
// MAESTRO DE / OLV -------------------
//58.[0OLV]=411;
// MAESTRO DE / ec-CASH ---------------
//58.[0CASH]=412;
// MAESTRO DE / default ---------------
//58.[0]=401;
// GELDKARTE --------------------------
//58.[30]=402;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
//58.[+67]=1;
// COMFORT CARD -----------------------
//58.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
//58.[+5]=2;
// DINERS -----------------------------
//58.[+36]=4;
// VISA -------------------------------
//58.[+4]=5;
// AMEX -------------------------------
//58.[+37]=6;
// JCB --------------------------------
//58.[+35]=3;
//
// **********************************************************************************************************************************************
// Terminal type 59 (AT PAYLIFE):
// The card recognition is based on the CARD NAME, given by the interface.
// (As default field CCC from p_belegzeile_1 or m_belegzeile_1, (supported by terminal software version 1.64 and greater) 
//  else m_name or, in the case of debit cards, the 3 first characters of p_text
//  else by card name)
// **********************************************************************************************************************************************
// MAESTRO -----------------------------
59.[d01]=1;
59.[d02]=1;
59.[edc]=1;
// MASTERCARD --------------------------
59.[d05]=2;
59.[c01]=2;
59.[c02]=2;
59.[KMC]=2;
59.[+mastercard]=2;
59.[+mc ]=2;
// JCB --------------------------------
59.[KJB]=3;
59.[d11]=3;
59.[c06]=3;
59.[+jcb]=3;
// DINERS -----------------------------
59.[d10]=4;
59.[c05]=4;
59.[KDI]=4;
59.[+diners]=4;
// VISA ELECTRON-----------------------
59.[+visa elec]=10;
// V PAY-------------------------------
59.[d07]=11;
59.[+v pay]=11;
59.[KVP]=11;
// VISA -------------------------------
59.[KVI]=5;
59.[d08]=5;
59.[c03]=5;
59.[+visa]=5;
59.[citibank visa]=5;
59.[hanseatic bank]=5;
59.[anzsmartpay]=5;
59.[ykb worldcard]=5;
// AMEX -------------------------------
59.[KAM]=6;
59.[d09]=6;
59.[c04]=6;
59.[+american]=6;
59.[+amex]=6;
// QUICK PIN based---------------------
59.[d06]=200;
59.[QCK]=200;
// BANKOMAT (MAESTRO AT) PIN based-----
59.[B-K]=201;
// BANKOMAT PIN based (not guaranteed)-
59.[BKN]=201;
59.[bkn]=201;
// JAPAN AIRLINES ---------------------
59.[+japan]=7;
// Euro Alliance of Payment Schemes----
59.[+eaps]=8;
// CUP (China UnionPay)----------------
59.[d12]=12;
59.[c07]=12;
59.[+cup]=12;
59.[+KCU]=12;
59.[+china]=12;
// FLOTTENKARTE (TYP 4)
59.[TYP4]=204,Flottenkarte;
// LOKALE KARTE (TYP 3)
59.[TYP3]=205;
// Inside Dornbirn
59.[V01OLP]=221,Inside Dornbirn;
59.[V01]=221,Inside Dornbirn;
59.[+Inside Dornbirn]=221,Inside Dornbirn;
//
// *******************************************************************************************************************************************
// Terminal type 60 (CH: Innocard Ingenico iPP350, iCT250 and iPP480 / ZVT):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *******************************************************************************************************************************************
// girocard ---------------------------
60.[5]=9,GIROCARD;
// EAPS -------------------------------
60.[7]=8,EAPS;
// Lastschrift ------
60.[9]=409,LASTSCHRIFT;
60.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
60.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
60.[13]=11,V PAY;
// BSW --------------------------------
60.[40]=404,BSW;
// POSTCARD ----------------------------
60.[69]=102,POSTFINANCE CARD;
60.[236]=102,POSTFINANCE CARD;
60.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
60.[46]=1;
// MASTERCARD -------------------------
60.[6]=2;
// DINERS -----------------------------
60.[12]=4;
// VISA -------------------------------
60.[10]=5;
// AMEX -------------------------------
60.[8]=6;
// JCB --------------------------------
60.[14]=3;
// HORNBACH Profi ---------------------
60.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
60.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
60.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
60.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
60.[42]=442,Comfort Card;
// Breuninger--------------------------
60.[38]=438,Breuninger;
60.[+352]=452,Paysafecard;
60.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
60.[23]=160,boncard POINTS;
60.[25]=161,boncard PAY;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
60.[218]=175;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
60.[+67]=1;
// COMFORT CARD -----------------------
60.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
60.[+5]=2;
60.[+222]=2;
60.[+223]=2;
60.[+224]=2;
60.[+225]=2;
60.[+226]=2;
60.[+227]=2;
60.[+228]=2;
60.[+229]=2;
60.[+230]=2;
60.[+231]=2;
60.[+232]=2;
60.[+233]=2;
60.[+234]=2;
60.[+235]=2;
60.[+236]=2;
60.[+237]=2;
60.[+238]=2;
60.[+239]=2;
60.[+240]=2;
60.[+241]=2;
60.[+242]=2;
60.[+243]=2;
60.[+244]=2;
60.[+245]=2;
60.[+246]=2;
60.[+247]=2;
60.[+248]=2;
60.[+249]=2;
60.[+250]=2;
60.[+251]=2;
60.[+252]=2;
60.[+253]=2;
60.[+254]=2;
60.[+255]=2;
60.[+256]=2;
60.[+257]=2;
60.[+258]=2;
60.[+259]=2;
60.[+261]=2;
60.[+262]=2;
60.[+263]=2;
60.[+264]=2;
60.[+265]=2;
60.[+266]=2;
60.[+267]=2;
60.[+268]=2;
60.[+269]=2;
60.[+270]=2;
60.[+271]=2;
60.[+272]=2;
// DINERS -----------------------------
60.[+36]=4;
// VISA -------------------------------
60.[+4]=5;
// AMEX -------------------------------
60.[+37]=6;
// JCB --------------------------------
60.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
60.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  -------------------------------
60.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
60.[+a0000001574480]=186;
// **********************************************************************************************************************************************
// Terminal type 61 (FR/ES/UK Ingenico Axis/C3 with callbacks):
// The card recognition is based on the cCardType (FFU: in combination with SSCard); fallback recognition by PAN  
// **********************************************************************************************************************************************
// AMEX ---------------------------------
61.[6]=6,AMERICAN EXPRESS;
// China Union Pay Natixis/China Union Pay 
61.[m]=12,CHINA UNION PAY;
// Diners Club de France ----------------
61.[7]=4,DINERS CLUB FRANCE;
// JCB piste/JCB EMV --------------------
61.[D]=3,JCB;
61.[+35]=3,JCB;
// MASTERCARD (by PAN)------------------
61.[+50]=2,MASTERCARD;
61.[+51]=2,MASTERCARD;
61.[+52]=2,MASTERCARD;
61.[+53]=2,MASTERCARD;
61.[+54]=2,MASTERCARD;
61.[+55]=2,MASTERCARD;
61.[+222]=2,MASTERCARD;
61.[+223]=2,MASTERCARD;
61.[+224]=2,MASTERCARD;
61.[+225]=2,MASTERCARD;
61.[+226]=2,MASTERCARD;
61.[+227]=2,MASTERCARD;
61.[+228]=2,MASTERCARD;
61.[+229]=2,MASTERCARD;
61.[+230]=2,MASTERCARD;
61.[+231]=2,MASTERCARD;
61.[+232]=2,MASTERCARD;
61.[+233]=2,MASTERCARD;
61.[+234]=2,MASTERCARD;
61.[+235]=2,MASTERCARD;
61.[+236]=2,MASTERCARD;
61.[+237]=2,MASTERCARD;
61.[+238]=2,MASTERCARD;
61.[+239]=2,MASTERCARD;
61.[+240]=2,MASTERCARD;
61.[+241]=2,MASTERCARD;
61.[+242]=2,MASTERCARD;
61.[+243]=2,MASTERCARD;
61.[+244]=2,MASTERCARD;
61.[+245]=2,MASTERCARD;
61.[+246]=2,MASTERCARD;
61.[+247]=2,MASTERCARD;
61.[+248]=2,MASTERCARD;
61.[+249]=2,MASTERCARD;
61.[+250]=2,MASTERCARD;
61.[+251]=2,MASTERCARD;
61.[+252]=2,MASTERCARD;
61.[+253]=2,MASTERCARD;
61.[+254]=2,MASTERCARD;
61.[+255]=2,MASTERCARD;
61.[+256]=2,MASTERCARD;
61.[+257]=2,MASTERCARD;
61.[+258]=2,MASTERCARD;
61.[+259]=2,MASTERCARD;
61.[+261]=2,MASTERCARD;
61.[+262]=2,MASTERCARD;
61.[+263]=2,MASTERCARD;
61.[+264]=2,MASTERCARD;
61.[+265]=2,MASTERCARD;
61.[+266]=2,MASTERCARD;
61.[+267]=2,MASTERCARD;
61.[+268]=2,MASTERCARD;
61.[+269]=2,MASTERCARD;
61.[+270]=2,MASTERCARD;
61.[+271]=2,MASTERCARD;
61.[+272]=2,MASTERCARD;
// MAESTRO (by PAN)---------------------
61.[+67]=1,MAESTRO;
61.[+63]=1,MAESTRO;
61.[+56]=1,MAESTRO;
61.[+57]=1,MAESTRO;
61.[+58]=1,MAESTRO;
61.[+59]=1,MAESTRO;
// VISA (by PAN)-----------------------
//61.[+4970]=900,CARTE BLEUE;
//61.[+4971]=900,CARTE BLEUE;
//61.[+4973]=900,CARTE BLEUE;
//61.[+4974]=900,CARTE BLEUE;
//61.[+4975]=900,CARTE BLEUE;
//61.[+4976]=900,CARTE BLEUE;
//61.[+4978]=900,CARTE BLEUE;
61.[+4]=5,VISA;
// ACCORD ------------------------------
61.[p]=901,ACCORD EMV;
61.[V]=901,ACCORD Piste; 
61.[w]=901,ACCORD sans contact;
// Alphyra (Payzone) --------------------
61.[k]=902,ALPHYRA;
// Cafeteria Casino ---------------------
61.[Q]=903,CAFETERIA CASINO;
// Auchan Telecom -----------------------
61.[u]=904,AUCHAN TELECOM;
// PNF Credit Mutuel --------------------
61.[W]=905,PNF CREDIT MUTUEL;
// Employe ED ---------------------------
61.[r]=906,EMPLOYE ED;
// EVEREST -------------------------------
61.[5]=907,EVEREST;
// COFINOGA -----------------------------
61.[l]=908,COFINOGA EMV; 
61.[2]=908,COFINOGA Piste et Sans Contact;
// Cheque -------------------------------
61.[M]=909,CHEQUE;
61.[N]=909,CHEQUE;
61.[O]=909,CHEQUE;
61.[P]=909,CHEQUE;
// Caledonienne de Service Bancaire ------
61.[d]=910,CALEDONIENNE DE SERVICE BANCAIRE;
// Reflexe/Aurore - protocole CETNCB CETDIS-
61.[8]=911,AURORE;
// Datavantage (GIFTCARD) ------------------
61.[i]=912,DATAVANTAGE;
// Edel - Leclerc --------------------------
61.[H]=913,EDEL;
// Total - parking -------------------------
61.[X]=914,ELF;
// Pluriel - FranFinance -------------------
61.[9]=915,FFI;
// Fnac ------------------------------------
61.[g]=916,FNAC; 
// Finaref ---------------------------------
61.[F]=917,FINAREF; 
// Gestion de Avoirs -----------------------
61.[z]=918,GAV; 
// Galerie Lafayette -----------------------
61.[c]=919,GLF; 
// Magellan --------------------------------
61.[y]=920,MGL;
// Multissime ------------------------------
61.[U]=921,MULTISSIME;
// PNF Optalion - credit Lyonnais ----------
61.[h]=922,OPTALION;
// Pass piste/Pass EMV ---------------------
61.[3]=923,PASS;
// Pegasus ---------------------------------
61.[v]=924,PGS;
// Moneo (Porte Monnaie Electronique) ------
61.[S]=925,PME;
// Promostim (Record Card) -----------------
61.[a]=926,PROMOSTIM;
// Prosodie --------------------------------
61.[j]=927,PROSODIE;
// Sofinco ---------------------------------
61.[B]=928,SOFINCO;
// Sodelem - carte petrole -----------------
61.[q]=929,SODELEM;
// Smiles ----------------------------------
61.[Y]=930,SMILES;
// **********************************************************************************************************************************************
// Terminal type 62 (BE ATOS NVIC 1.07/13):
// The card recognition is based on the BRAND ID, on the BIT MAP OF APPLICATION ID or on the CARDNAME (VIC 1.07/13).
// *******************************************************************************************************************************************
// VISA Electron ----------------------
62.[114]=10;
62.[1002]=10;
62.[visa electron]=10;
// VISA -------------------------------
62.[124]=5;
62.[2002]=5;
62.[visa]=5;
// VISA VPay---------------------------
62.[82]=10;
62.[+vpay]=10;
// JCB --------------------------------
62.[120]=3;
62.[2007]=3;
62.[jcb]=3;
// DINERS -----------------------------
62.[121]=4;
62.[2005]=4;
62.[diners]=4;
// AMEX -------------------------------
62.[122]=6;
62.[2004]=6;
62.[american express]=6;
// MASTERCARD -------------------------
62.[123]=2;
62.[2003]=2;
62.[mastercard]=2;
// Maestro ----------------------------
62.[127]=1;
62.[1009]=1;
62.[maestro]=1;
// Bancontact/Mister Cash -------------
62.[128]=500;
62.[1001]=500;
62.[+bancontact]=500;
62.[66]=500;
62.[1013]=500;
62.[tina]=500,TINA;
62.[100]=500,TINA;
// PROTON -----------------------------
62.[125]=501;
62.[1010]=501;
62.[proton]=501;
// AURORA -----------------------------
62.[109]=502;
62.[3001]=502;
62.[aurora]=502;
// KADOZ ------------------------------
62.[64]=503;
62.[kadoz]=503;
// EDENRED ----------------------------
62.[69]=504;
62.[3004]=504;
62.[edenred]=504;
// SODEXO EPASS------------------------
62.[sodexo epass]=505;
62.[70]=505;
62.[3003]=505;
// MONIZZE------------------------
62.[2009]=506;
62.[71]=506;
62.[monizze]=506;
// TINA EMV ---------------------------
62.[tina emv]=TINA EMV;
62.[65]=TINA EMV;
// CHINA UNION PAY --------------------
62.[2008]=12;
// JEWEL CARD ---------------------------
62.[3300]=720;
62.[60]=720;
62.[Giftcard]=720;
// **********************************************************************************************************************************************
// Terminal type 63 (NL ATOS NVIC 1.07/13):
// The card recognition is based on the BRAND ID, on the BIT MAP OF APPLICATION ID or on the CARDNAME (VIC 1.07/13).
// *******************************************************************************************************************************************
// VISA Electron ----------------------
63.[114]=10;
63.[1002]=10;
63.[visa electron]=10;
// VISA -------------------------------
63.[124]=5;
63.[2002]=5;
63.[visa]=5;
// VISA VPay---------------------------
63.[82]=10;
63.[+vpay]=10;
// JCB --------------------------------
63.[120]=3;
63.[2007]=3;
63.[jcb]=3;
// DINERS -----------------------------
63.[121]=4;
63.[2005]=4;
63.[diners]=4;
// AMEX -------------------------------
63.[122]=6;
63.[2004]=6;
63.[american express]=6;
// MASTERCARD -------------------------
63.[123]=2;
63.[2003]=2;
63.[mastercard]=2;
// Maestro ----------------------------
63.[127]=1;
63.[1009]=1;
63.[maestro]=1;
// Bancontact/Mister Cash -------------
63.[128]=500;
63.[1001]=500;
63.[+bancontact]=500;
63.[66]=500;
63.[1013]=500;
63.[tina]=500,TINA;
63.[100]=500,TINA;
// PROTON -----------------------------
63.[125]=501;
63.[1010]=501;
63.[proton]=501;
// AURORA -----------------------------
63.[109]=502;
63.[3001]=502;
63.[aurora]=502;
// KADOZ ------------------------------
63.[64]=503;
63.[kadoz]=503;
// EDENRED ----------------------------
63.[69]=504;
63.[3004]=504;
63.[edenred]=504;
// SODEXO EPASS------------------------
63.[sodexo epass]=505;
63.[70]=505;
63.[3003]=505;
// MONIZZE------------------------
63.[2009]=506;
63.[71]=506;
63.[monizze]=506;
// TINA EMV ---------------------------
63.[tina emv]=TINA EMV;
63.[65]=TINA EMV;
// CHINA UNION PAY --------------------
63.[2008]=12;
// JEWEL CARD ---------------------------
63.[3300]=720;
63.[60]=720;
63.[Giftcard]=720;
// **********************************************************************************************************************************************
// Terminal type 64 (LU ATOS NVIC 1.07/13):
// The card recognition is based on the BRAND ID, on the BIT MAP OF APPLICATION ID or on the CARDNAME (VIC 1.07/13).
// *******************************************************************************************************************************************
// VISA Electron ----------------------
64.[114]=10;
64.[1002]=10;
64.[visa electron]=10;
// VISA -------------------------------
64.[124]=5;
64.[2002]=5;
64.[visa]=5;
// VISA VPay---------------------------
64.[82]=10;
64.[+vpay]=10;
// JCB --------------------------------
64.[120]=3;
64.[2007]=3;
64.[jcb]=3;
// DINERS -----------------------------
64.[121]=4;
64.[2005]=4;
64.[diners]=4;
// AMEX -------------------------------
64.[122]=6;
64.[2004]=6;
64.[american express]=6;
// MASTERCARD -------------------------
64.[123]=2;
64.[2003]=2;
64.[mastercard]=2;
// Maestro ----------------------------
64.[127]=1;
64.[1009]=1;
64.[maestro]=1;
// Bancontact/Mister Cash -------------
64.[128]=500;
64.[1001]=500;
64.[+bancontact]=500;
64.[66]=500;
64.[1013]=500;
64.[tina]=500,TINA;
64.[100]=500,TINA;
// PROTON -----------------------------
64.[125]=501;
64.[1010]=501;
64.[proton]=501;
// AURORA -----------------------------
64.[109]=502;
64.[3001]=502;
64.[aurora]=502;
// KADOZ ------------------------------
64.[64]=503;
64.[kadoz]=503;
// EDENRED ----------------------------
64.[69]=504;
64.[3004]=504;
64.[edenred]=504;
// SODEXO EPASS------------------------
64.[sodexo epass]=505;
64.[70]=505;
64.[3003]=505;
// MONIZZE------------------------
64.[2009]=506;
64.[71]=506;
64.[monizze]=506;
// TINA EMV ---------------------------
64.[tina emv]=TINA EMV;
64.[65]=TINA EMV;
// CHINA UNION PAY --------------------
64.[2008]=12;
// JEWEL CARD ---------------------------
64.[3300]=720;
64.[60]=720;
64.[Giftcard]=720;
// **********************************************************************************************************************************************
// Terminal type 65 (NL BANKSYS / VIC Standard oriented YOMANI 1.06A (Draft)):
// The card recognition is based on mod_name (Bit 180).
// *******************************************************************************************************************************************
// VISA -------------------------------
65.[VISA]=5;
65.[Visa]=5;
65.[V Pay]=11;
// JCB --------------------------------
65.[JCB]=3;
// DINERS -----------------------------
65.[DINERS]=4;
65.[Diners]=4;
// AMEX -------------------------------
65.[AMEX]=6;
// MASTERCARD -------------------------
65.[ECMC]=2;
65.[MasterCard]=2;
// Maestro ----------------------------
65.[MAES]=1;
65.[Maestro]=1;
// Bancontact/Mister Cash -------------
65.[BANK]=700;
// Postbank ---------------------------
65.[GIRO]=9;
// EPWS --------------------------------
65.[EPWS]=702;
// CUP ---------------------------------
65.[CUP]=12;
// COMFORTCARD -------------------------
65.[CMFC]=704;
65.[COMF]=704;
// PRIMELINE----------------------------
65.[PRLI]=705;
65.[PRIM]=705;
// Basic Card --------------------------
65.[Basic Card]=706;
// Schoenenbon -------------------------
65.[Schoenenbon]=707;
// Fashion Card ------------------------
65.[Fashion Card]=708;
// CCV Card ----------------------------
65.[CCV Card]=709;
// Yourgift ----------------------------
65.[Yourgift H&G]=710;
65.[Yourgift G&M]=711;
65.[Yourgift M&M]=712;
65.[Yourgift B&H]=713;
65.[Yourgift H&T]=714;
65.[Yourgift F&F]=715;
65.[Yourgift W&D]=716;
65.[Yourgift G&C]=717;
// HORNBACH PROFICARD NL---------------
65.[HORN]=718;
65.[CCV]=718;
// Chipknip ----------------------------
65.[CHIP]=719;
//
// **********************************************************************************************************************************************
// Terminal type 66 (SK WNSP Ingenico ):
// The card recognition is based on Issuer Name
// **********************************************************************************************************************************************
// HORNBACH ---------------------------
66.[+hornbach]=420;
// AMEX -------------------------------
66.[+amex]=6;
66.[+american]=6;
// DCCB - Diners Club -----------------
66.[+diners]=4;
66.[+dci]=4;
// MASTERCARD -------------------------     
66.[+ec/mc]=2;
66.[+master]=2;
// MAESTRO ----------------------------
66.[+maestro]=1;
// VISA -------------------------------
66.[+visa]=5;
// JCB --------------------------------
66.[+jcb]=3;
66.[+japan]=3;
//
// **********************************************************************************************************************************************
// Terminal type 67 (RO WNSP Ingenico ):
// The card recognition is based on Issuer Name
// **********************************************************************************************************************************************
// HORNBACH ---------------------------
67.[+hornbach]=420;
// AMEX -------------------------------
67.[+amex]=6;
67.[+american]=6;
// DCCB - Diners Club -----------------
67.[+diners]=4;
67.[+dci]=4;
// MASTERCARD -------------------------     
67.[+ec/mc]=2;
67.[+master]=2;
// MAESTRO ----------------------------
67.[+maestro]=1;
// VISA -------------------------------
67.[+visa]=5;
// JCB --------------------------------
67.[+jcb]=3;
67.[+japan]=3;
//
// **********************************************************************************************************************************************
// Terminal type 68 (CH VIBBEK): 
// The card recognition is based on AID and card name
// **********************************************************************************************************************************************
// MAESTRO CH --------------------
68.[+a0000000101030]=100;
68.[+a0000001570022]=100;
68.[a0000000031030]=100;
68.[+maestro-ch]=100;
68.[+maestro ch]=100;
// MAESTRO ------------------------
68.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
68.[+a0000001570021]=1;
// MASTERCARD ---------------------
68.[+a0000000041010]=2;
68.[+a0000001570020]=2;
// M-Budget -----------------------
68.[a0000000000008]=2;
// JCB ---------------------------
68.[+a0000000651010]=3;
68.[+a0000001570040]=3;
// DINERS ------------------------
68.[+a0000001884443]=4;
68.[+a0000001574443]=4;
68.[+a0000001570060]=4;
68.[+a0000001523010]=4;
// VISA --------------------------
68.[+a0000000031010]=5;
68.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
68.[+a0000000032010]=10;
68.[+a0000001570031]=10;
// V PAY -------------------------------
68.[+a0000000032020]=11,V PAY;
68.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
68.[+a0000001570010]=6;
68.[+a00000002501]=6;
68.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
68.[a000000157446003]=12,CHINA UNION PAY;
68.[a000000157446004]=12,CHINA UNION PAY;
68.[a000000157446005]=12,CHINA UNION PAY;
68.[a000000157446006]=12,CHINA UNION PAY;
68.[a000000157446009]=12,CHINA UNION PAY;
68.[a0000001574460]=12,CHINA UNION PAY;
68.[a000000333010101]=12,CHINA UNION PAY;
68.[a000000333010102]=12,CHINA UNION PAY;
68.[a000000333010103]=12,CHINA UNION PAY;
// M-CARD ------------------------
68.[a0000001570100]=101;
68.[a0000001574450]=101;
// POSTCARD
68.[+a0000001110101]=102;
68.[+a0000001999999]=102;
68.[+a0000001999998]=102;
68.[+a0000001570051]=102;
68.[+a0000001570050]=102;
68.[+a0000001574445]=102;
68.[+a0000001574446]=102;
// BERN CARD --------------------
68.[+a0000001570107]=103;
// ZUERI CARD -------------------
68.[+a0000001570106]=104;
// BASLER CARD ------------------
68.[+a0000001570108]=105;
// CASH -------------------------
68.[+a0000001570023]=106;
// PHARMACARD --------------------
68.[+a0000001570105]=107;
// MOEVENPICK -------------------
68.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
68.[+a0000001570103]=110;
68.[+a000000157010B]=110;
68.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
68.[a0000001574462]=137;
// MY ONE -----------------------
68.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
68.[a0000001574444]=116;
68.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
68.[a0000001574458]=134;
68.[a00000000300037562]=134;
68.[a0000000037562]=134;
// POWER CARD -------------------
68.[+a000000157010D]=117;
68.[a0000001574476]=117;
68.[a0000001574478]=117;
68.[a0000001574479]=117;
// WIR --------------------------
68.[+a000000157010C]=118;
// REKA  ------------------------------
68.[d7560000011111]=120;
68.[+a000000157010E]=120;
68.[+D7560000010101]=120;
// REKA RAIL --------------------------
68.[d7560000011112]=180;
68.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
68.[d7560000011113]=178;
68.[d7560000010102]=178;
68.[+a0000001574497]=178
// REKA E-COMMERCE --------------------
68.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
68.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
68.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
68.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
68.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
68.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
68.[a0000001574455]=160,BONARD POINTS;
68.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK
68.[a000000157447d]=172;
68.[d7560001150001]=172;
// CCV PAY
68.[a000000157447b]=173;
// COOP EKZ
68.[37]=174;
68.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
68.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
68.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
68.[a0000001574482]=177;
// GROUPCARD (Loyalty Gift Card AG)------
68.[+a0000001574480]=186;
// TWINT -------------------------------
68.[a000000157449e]=184,TWINT;
// SWISS PAY
68.[a0000008010001]=2000,SWISS PAY;
68.[37]=2000,SWISS PAY;
// SWISS PAY 2
68.[a00000015744BD]=2001,SWISS PAY 2;
68.[37]=2001,SWISS PAY 2;
// **********************************************************************************************************************************************
// Terminal type 69 (CA/US AJB FIPAY):
// The card recognition is based on Card Type Name
// **********************************************************************************************************************************************
// MASTERCARD -------------------------     
69.[+mastercard]=2;
// JCB ---------------------------
69.[+jcb]=3;
// DCCB - Diners Club -----------------
69.[+diners]=4;
69.[+ds]=4;
// VISA -------------------------------
69.[+visa]=5;
// AMEX -------------------------------
69.[+amex]=6;
// CHINA UNION PAY --------------------
69.[+cup]=12;
// DISCOVER --------------------------------
69.[+discover]=600;
// DEBIT
69.[+debit]=13;
//
// **********************************************************************************************************************************************
// Terminal type 70 (UK GLOBAL PAYMENTS EPOS INTERFACE): 
// The card recognition is based on the AID and on the ISSUER NAME
// **********************************************************************************************************************************************
// MAESTRO ------------------------
70.[+a0000000043060]=1;
70.[+a0000001570021]=1;
70.[+maestro]=1;
// MASTERCARD ---------------------
70.[+a0000000041010]=2;
70.[+a0000001570020]=2;
70.[+mastercard]=2;
// JCB ---------------------------
70.[+a0000000651010]=3;
70.[+a0000001570040]=3;
70.[+jcb]=3;
// DINERS ------------------------
70.[+a0000001884443]=4;
70.[+a0000001574443]=4;
70.[+a0000001570060]=4;
70.[+a0000001523010]=4;
70.[+diners]=4;
// VISA --------------------------
70.[+a0000001570030]=5;
70.[+visa]=5;
70.[+a0000000032010]=5;
70.[+a0000001570031]=5;
70.[+a0000000031010]=5;
70.[+visa debit]=5;
70.[+a0000000032020]=5;
70.[a0000001574452]=5;
70.[+VISA Credit]=5;
// AMERICAN EXPRESS --------------
70.[+a0000001570010]=6;
70.[+a00000002501]=6;
70.[+a0000000291010]=6;
70.[+amex]=6;
70.[+american]=6;
// CHINA UNION PAY --------------------
70.[a000000157446003]=12,CHINA UNION PAY;
70.[a000000157446004]=12,CHINA UNION PAY;
70.[a000000157446005]=12,CHINA UNION PAY;
70.[a000000157446006]=12,CHINA UNION PAY;
70.[a000000157446009]=12,CHINA UNION PAY;
70.[a0000001574460]=12,CHINA UNION PAY;
70.[a000000333010101]=12,CHINA UNION PAY;
70.[a000000333010102]=12,CHINA UNION PAY;
70.[+cup]=12,CHINA UNION PAY;
70.[+unionpay]=12,CHINA UNION PAY;
70.[+union pay]=12,CHINA UNION PAY;
70.[+chinaunionpay]=12,CHINA UNION PAY;
//
// **********************************************************************************************************************************************
// Terminal type 71 (AU PC-EFTPOS): 
// The card recognition is based on cardType
// **********************************************************************************************************************************************
// DEBIT
71.[01]=13;
// MASTERCARD -------------------------     
71.[03]=2;
// VISA -------------------------------
71.[04]=5;
// AMEX -------------------------------
71.[05]=6;
// DCCB - Diners Club -----------------
71.[06]=4;
// JCB ---------------------------
71.[07]=3;
71.[09]=3;
71.[11]=3;
// CHINA UNION PAY --------------------
71.[02]=12;
//
// **********************************************************************************************************************************************
// Terminal type 72 (CH SIX CCREDIT): 
// The card recognition is based on the ccredit:CardType and the CardCircuit
// **********************************************************************************************************************************************
// AMERICAN EXPRESS --------------
72.[+american]=6,AMERICAN EXPRESS;
72.[+amex]=6,AMERICAN EXPRESS;
72.[1]=6,AMERICAN EXPRESS;
// MASTERCARD ---------------------
72.[+master]=2,MASTERCARD;
72.[2]=2,MASTERCARD;
// VISA ---------------------------
72.[3]=5,VISA;
72.[+visa]=5,VISA;
72.[57]=5,VISA;
72.[59]=5,VISA;
// DINERS ------------------------
72.[+diners]=4,DINERS;
72.[4]=4,DINERS;
// JCB ---------------------------
72.[+jcb]=3,JCB;
72.[5]=3,JCB;
// MAESTRO ------------------------
72.[+maestro]=1,MAESTRO;
72.[6]=1,MAESTRO;
72.[12]=1,MAESTRO;
72.[61]=1,MAESTRO;
72.[62]=1,MAESTRO;
// GIROCARD ---------------------------
72.[7]=9,GIROCARD;
72.[+ec ]=9,GIROCARD;
72.[+giro]=9,GIROCARD;
// VISA ELECTRON ----------------------
72.[49]=10,VISA ELECTRON;
72.[58]=10,VISA ELECTRON;
// V PAY ------------------------------
72.[52]=11,V PAY;
72.[v pay]=11,V PAY;
72.[v-pay]=11,V PAY;
// POSTCARD ---------------------------
72.[8]=102,POSTFINANCE CARD;
72.[+post]=102,POSTFINANCE CARD;
// IKEA-CARD---------------------------
72.[9]=119,IKEA-CARD;
72.[+ikea]=119,IKEA-CARD;
// Breuninger-- -----------------------
72.[10]=438,BREUNINGER;
72.[+breuninger]=438,BREUNINGER;
// JELMOLI BONUS SHOPPING CARD --------
72.[11]=110,JELMOLI CARD;
72.[+jelmoli]=110,JELMOLI CARD
// QUICK PIN based---------------------
72.[+quick]=200,QUICK;
72.[13]=200,QUICK;
// MYONE ------------------------------
72.[15]=111,MYONE;
72.[+myone]=111,MYONE;
// POWERCARD --------------------------
72.[16]=117,POWERCARD;
72.[+power]=117,POWERCARD;
// REKA  -------------------------------
72.[17]=120,REKA;
72.[+reka]=120,REKA;
72.[90]=120,REKA;
// PRIMELINE ---------------------------
72.[19]=705,PRIMELINE;
72.[+prime]=705,PRIMELINE;
// COMFORT CARD -----------------------
72.[20]=442,COMFORT CARD;
72.[+comfort]=442,COMFORT CARD;
// CHIPKNIP ----------------------------
72.[23]=719,CHIPKNIP;
72.[+chipknip]=719,CHIPKNIP;
// O2 LOOP ----------------------------
72.[24]=481,O2 LOOP;
72.[+o2]=481,O2 LOOP;
72.[+loop]=481,O2 LOOP;
// E-PLUS -----------------------------
72.[25]=480,E-PLUS FREE&EASY;
72.[+E-Plus Free&Easy]=480,E-PLUS FREE&EASY;
72.[+E-Plus]=480,E-PLUS FREE&EASY;
72.[+Free&Easy]=480,E-PLUS FREE&EASY;
// T-D1 XTRA --------------------------
72.[29]=441,T-D1 XTRA;
72.[+T D1]=441,T-D1 XTRA;
72.[+T-D1]=441,T-D1 XTRA;
72.[+XTRA]=441,T-D1 XTRA;
// VODAFONE CALLYA --------------------
72.[30]=479,VODAFONE CALLYA;
72.[+Vodafone]=479,VODAFONE CALLYA;
72.[+CallYa]=479,VODAFONE CALLYA;
// ADUNO CARDS --------------
72.[31]=124,SWISS BONUS CARD;
72.[+swiss bonus]=124,SWISS BONUS CARD;
72.[32]=125,ABO CARD;
72.[+abo card]=125,ABO CARD;
// PHARMACARD --------------------
72.[33]=107,PHARMACARD;
72.[+pharma]=107,PHARMACARD;
// M-CARD -----------------------------
72.[34]=101,M-CARD;
72.[m-card]=101,M-CARD;
72.[m card]=101,M-CARD;
// YOURGIFT ----------------------------
72.[37]=710,YOURGIFT;
72.[Yourgift H&G]=710,YOURGIFT H&G;
72.[Yourgift G&M]=711,YOURGIFT G&M;
72.[Yourgift M&M]=712,YOURGIFT M&M;
72.[Yourgift B&H]=713,YOURGIFT B&H;
72.[Yourgift H&T]=714,YOURGIFT H&T;
72.[Yourgift F&F]=715,YOURGIFT F&F;
72.[Yourgift W&D]=716,YOURGIFT W&D;
72.[Yourgift G&C]=717,YOURGIFT G&C;
72.[+yourgift]=710,YOURGIFT;
// BASIC CARD --------------------------
72.[38]=706,BASIC CARD;
72.[basic card]=706,BASIC CARD;
// FASHION CARD ------------------------
72.[40]=708,FASHION CARD;
72.[fashion card]=708,FASHION CARD;
// SCHOENENBON -------------------------
72.[41]=707,SCHOENENBON;
72.[schoenenbon]=707,SCHOENENBON;
// CCV CARD ----------------------------
72.[42]=709,CCV CARD;
72.[ccv card]=709,CCV CARD;
// CUP (CHINA UNION PAY)----------------
72.[43]=12,CHINA UNION PAY;
72.[cup]=12,CHINA UNION PAY;
72.[+china]=12,CHINA UNION PAY;
// BACONTACT/MISTER CASH-------------
72.[46]=500,BANCOTACT/MCA;
72.[+bancontact]=500,BANCOTACT/MCA;
// PROTON -----------------------------
72.[47]=501,PROTON;
72.[+proton]=501,PROTON;
// AURORA -----------------------------
72.[48]=502,AURORA;
72.[+aurora]=502,AURORA;
// DANKORT ----------------------------
72.[50]=1100,DANKORT;
72.[+dankort]=1100,DANKORT;
// SWISSCADEAU --------------------------
72.[66]=136,SWISSCADEAU;
72.[+swisscadeau]=136,SWISSCADEAU;
// GROUPCARD (Loyalty Gift Card AG)------
72.[+a0000001574480]=186;
// ALIPAY ---------------------------
72.[a000000162000503]=17,AliPay;
72.[a00000015744A0]=17,AliPay;
72.[+alipay]=17,AliPay;
72.[78]=17,AliPay;
//
// **********************************************************************************************************************************************
// Terminal type 73 (HK Pax Interface for credit and cup / eps cards): 
// The card recognition is based on the AID and the card name
// **********************************************************************************************************************************************
// MAESTRO ------------------------
73.[+maestro]=1;
73.[+a0000000043060]=1;
73.[+a0000001570021]=1;
// MASTERCARD ---------------------
73.[+master]=2;
73.[+mc]=2,MASTERCARD;
73.[+a0000000041010]=2;
73.[+a0000001570020]=2;
// JCB ---------------------------
73.[+jcb]=3;
73.[+a0000000651010]=3;
73.[+a0000001570040]=3;
// DINERS ------------------------
73.[+diners]=4;
73.[+a0000001884443]=4;
73.[+a0000001574443]=4;
73.[+a0000001570060]=4;
73.[+a0000001523010]=4;
// VISA --------------------------
73.[+visa]=5;
73.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
73.[+a0000000032010]=10;
73.[+a0000001570031]=10;
// V PAY (this will be routed to VISA after consenting with global pay) -------------------------------
73.[+a0000000031010]=5;
73.[+a0000000032020]=5;
73.[a0000001574452]=5;
73.[+visa debit]=5,VISA;
// AMERICAN EXPRESS --------------
73.[+amex]=6;
73.[+american express]=6;
73.[+a0000001570010]=6;
73.[+a00000002501]=6;
73.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
73.[+cup]=12;
73.[+china union pay]=12;
73.[+unionpay]=12;
73.[a000000157446003]=12,CHINA UNION PAY;
73.[a000000157446004]=12,CHINA UNION PAY;
73.[a000000157446005]=12,CHINA UNION PAY;
73.[a000000157446006]=12,CHINA UNION PAY;
73.[a000000157446009]=12,CHINA UNION PAY;
73.[a0000001574460]=12,CHINA UNION PAY;
73.[a000000333010101]=12,CHINA UNION PAY;
73.[a000000333010102]=12,CHINA UNION PAY;
73.[a000000333010102]=12,CHINA UNION PAY;
// EPS (ELECTRONIC PAYMENTS SERVICES) --------------------
73.[+eps]=1400,EPS; 
// MAESTRO CH --------------------
73.[+a0000000101030]=100;
73.[+a0000001570022]=100;
73.[a0000000031030]=100;
// M-CARD ------------------------
73.[a0000001570100]=101;
73.[a0000001574450]=101;
// POSTCARD
73.[+a0000001110101]=102;
73.[+a0000001999999]=102;
73.[+a0000001999998]=102;
73.[+a0000001570051]=102;
73.[+a0000001570050]=102;
73.[+a0000001574445]=102;
73.[+a0000001574446]=102;
// **********************************************************************************************************************************************
// Terminal type 74 (SE VERIFONE/POINT): 
// The card recognition is based on the AID and Card Name
// **********************************************************************************************************************************************
// MAESTRO ------------------------
74.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
74.[+a0000001570021]=1;
74.[+maestro]=1;
// MASTERCARD ---------------------
74.[+a0000000041010]=2;
74.[+a0000001570020]=2;
74.[+mastercard]=2;
// JCB ---------------------------
74.[+a0000000651010]=3;
74.[+a0000001570040]=3;
// DINERS ------------------------
74.[+a0000001884443]=4;
74.[+a0000001574443]=4;
74.[+a0000001570060]=4;
74.[+a0000001523010]=4;
// VISA --------------------------
74.[+a0000000031010]=5;
74.[+a0000001570030]=5;
74.[+visa]=5;
// VISA ELECTRON-----------------------
74.[+a0000000032010]=10;
74.[+a0000001570031]=10;
74.[+visa elec]=10;
// V PAY -------------------------------
74.[+a0000000032020]=11,V PAY;
74.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
74.[+a0000001570010]=6;
74.[+a00000002501]=6;
74.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
74.[a000000157446003]=12,CHINA UNION PAY;
74.[a000000157446004]=12,CHINA UNION PAY;
74.[a000000157446005]=12,CHINA UNION PAY;
74.[a000000157446006]=12,CHINA UNION PAY;
74.[a000000157446009]=12,CHINA UNION PAY;
74.[a0000001574460]=12,CHINA UNION PAY;
74.[a000000333010101]=12,CHINA UNION PAY;
74.[a000000333010102]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
74.[+a0000000101030]=100;
74.[+a0000001570022]=100;
74.[a0000000031030]=100;
74.[+maestro ch]=100;
// Hornbach-Kundenkarten ---------------------
74.[+PROFIC]=1116,PROFFS KORT;
74.[+PROJEK]=1117,PROJEKTVÄRLDSKORT;
// **********************************************************************************************************************************************
// Terminal type 75 (Panasonic iNFOX ZEC-14 and JT-C16 Series):
// The card recognition is based on the CARD NUMBER PREFIX and CARD COMPANY NUMBER (KID).
// **********************************************************************************************************************************************
// MASTERCARD --------------------------
75.[+2720]=2,MASTERCARD;
75.[+271]=2,MASTERCARD;
75.[+270]=2,MASTERCARD;
75.[+26]=2,MASTERCARD;
75.[+25]=2,MASTERCARD;
75.[+24]=2,MASTERCARD;
75.[+23]=2,MASTERCARD;
75.[+229]=2,MASTERCARD;
75.[+228]=2,MASTERCARD;
75.[+227]=2,MASTERCARD;
75.[+226]=2,MASTERCARD;
75.[+225]=2,MASTERCARD;
75.[+224]=2,MASTERCARD;
75.[+223]=2,MASTERCARD;
75.[+2229]=2,MASTERCARD;
75.[+2228]=2,MASTERCARD;
75.[+2227]=2,MASTERCARD;
75.[+2226]=2,MASTERCARD;
75.[+2225]=2,MASTERCARD;
75.[+2224]=2,MASTERCARD;
75.[+2223]=2,MASTERCARD;
75.[+2222]=2,MASTERCARD;
75.[+2221]=2,MASTERCARD;
75.[+55]=2,MASTERCARD;
75.[+54]=2,MASTERCARD;
75.[+53]=2,MASTERCARD;
75.[+52]=2,MASTERCARD;
75.[+51]=2,MASTERCARD;
// JCB --------------------------------
75.[102]=3,JCB;
75.[+358]=3,JCB;
75.[+357]=3,JCB;
75.[+356]=3,JCB;
75.[+355]=3,JCB;
75.[+354]=3,JCB;
75.[+353]=3,JCB;
75.[+3529]=3,JCB;
75.[+3528]=3,JCB;
// DINERS -----------------------------
75.[101]=4,DINERS;
75.[+309]=4,DINERS;
75.[+305]=4,DINERS;
75.[+304]=4,DINERS;
75.[+303]=4,DINERS;
75.[+302]=4,DINERS;
75.[+301]=4,DINERS;
75.[+300]=4,DINERS;
75.[+39]=4,DINERS;
75.[+38]=4,DINERS;
75.[+36]=4,DINERS;
// VISA -------------------------------
75.[104]=5,VISA;
75.[+4]=5,VISA;
// AMEX -------------------------------
75.[107]=6,AMEX;
75.[+37]=6,AMEX;
75.[+34]=6,AMEX;
// J-DEBIT ---------------------------
75.[J-DEBIT]=1201,J-DEBIT;
// CUP -------------------------------
75.[CUP]=12,CUP;
// **********************************************************************************************************************************************
// Terminal type 76 (DE Verifone MX925):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// *******************************************************************************************************************************************
// CHINA UNION PAY ---------------------
76.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
76.[5]=9,GIROCARD;
// EAPS -------------------------------
76.[7]=8,EAPS;
// Lastschrift ------
76.[9]=409,LASTSCHRIFT;
76.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
76.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
76.[13]=11,V PAY;
// BSW --------------------------------
76.[40]=404,BSW;
// MAESTRO ----------------------------
76.[46]=1;
// MASTERCARD -------------------------
76.[6]=2;
// DINERS -----------------------------
76.[12]=4;
// VISA -------------------------------
76.[10]=5;
// AMEX -------------------------------
76.[8]=6;
// JCB --------------------------------
76.[14]=3;
// MAESTRO DE / ELV -------------------
76.[2ELV]=410;
// MAESTRO DE / OLV -------------------
76.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
76.[2CASH]=412;
// MAESTRO DE / default ---------------
76.[2]=401;
// GELDKARTE --------------------------
76.[30]=402;
// HORNBACH Profi ---------------------
76.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
76.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
76.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
76.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
76.[42]=442,Comfort Card;
// Breuninger -------------------------
76.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
76.[+414]=414,AnniCard;
76.[+AnniCard]=414,AnniCard;
//
76.[212]=441,T-D1 XTRA;
76.[80]=441,T-D1 XTRA;
76.[+T-D1 XTRA]=441,T-D1 XTRA;
76.[+T-D1]=441,T-D1 XTRA;
76.[+XTRA]=441,T-D1 XTRA;
//
76.[191]=479,Vodafone CallYa;
76.[+Vodafone CallYa]=479,Vodafone CallYa;
76.[+Vodafone]=479,Vodafone CallYa;
76.[+CallYa]=479,Vodafone CallYa;
//
76.[190]=480,E-Plus Free&Easy;
76.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
76.[+E-Plus]=480,E-Plus Free&Easy;
76.[+Free&Easy]=480,E-Plus Free&Easy;
//
76.[188]=481,O2 LOOP;
76.[+O2 LOOP]=481,O2 LOOP;
76.[+O2]=481,O2 LOOP;
76.[+LOOP]=481,O2 LOOP;
//
76.[+431]=431,Premiere Flex;
76.[+Premiere Flex]=431,Premiere Flex;
76.[+Premiere]=431,Premiere Flex;
76.[+Flex]=431,Premiere Flex;
//
76.[189]=405,gobananas;
76.[+gobananas]=405,gobananas;
//
76.[180]=452,Paysafecard Blue;
76.[181]=452,Paysafecard Red;
76.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
76.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
76.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
76.[807]=484,SCHLECKER PREPAID;
76.[249]=486,smobil;
76.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
76.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
76.[246]=488,FONIC;
76.[206]=489,BILDmobil;
76.[205]=490,Blau.de;
76.[207]=491,Congstar;
76.[475]=492,Lebara Mobile;
76.[248]=493,Simyo;
76.[884]=494,Lycamobile;
76.[917]=495,GT mobile;
// AVS -----------------------------
76.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
76.[+6010822]=497,Alnatura Gutscheinkarte;
// 220 TLAPA-KARTE---------------------
76.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
76.[+67]=1;
// COMFORT CARD -----------------------
76.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
76.[+5]=2;
76.[+222]=2;
76.[+223]=2;
76.[+224]=2;
76.[+225]=2;
76.[+226]=2;
76.[+227]=2;
76.[+228]=2;
76.[+229]=2;
76.[+230]=2;
76.[+231]=2;
76.[+232]=2;
76.[+233]=2;
76.[+234]=2;
76.[+235]=2;
76.[+236]=2;
76.[+237]=2;
76.[+238]=2;
76.[+239]=2;
76.[+240]=2;
76.[+241]=2;
76.[+242]=2;
76.[+243]=2;
76.[+244]=2;
76.[+245]=2;
76.[+246]=2;
76.[+247]=2;
76.[+248]=2;
76.[+249]=2;
76.[+250]=2;
76.[+251]=2;
76.[+252]=2;
76.[+253]=2;
76.[+254]=2;
76.[+255]=2;
76.[+256]=2;
76.[+257]=2;
76.[+258]=2;
76.[+259]=2;
76.[+261]=2;
76.[+262]=2;
76.[+263]=2;
76.[+264]=2;
76.[+265]=2;
76.[+266]=2;
76.[+267]=2;
76.[+268]=2;
76.[+269]=2;
76.[+270]=2;
76.[+271]=2;
76.[+272]=2;
// DINERS -----------------------------
76.[+36]=4;
// VISA -------------------------------
76.[+4]=5;
// AMEX -------------------------------
76.[+37]=6;
// JCB --------------------------------
76.[+35]=3;
// Alipay ---------------------
76.[97]=17,AliPay;
// Bluecode --------------------------------
76.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 77 (Verifone VX520 Hypercom VTINT protocol Thailand):
// The card recognition is based on the AID and on the CARD NUMBER PREFIX.
// **********************************************************************************************************************************************
// MASTERCARD --------------------------
77.[+a0000000041010]=2,MASTERCARD;
77.[+2720]=2,MASTERCARD;
77.[+271]=2,MASTERCARD;
77.[+270]=2,MASTERCARD;
77.[+26]=2,MASTERCARD;
77.[+25]=2,MASTERCARD;
77.[+24]=2,MASTERCARD;
77.[+23]=2,MASTERCARD;
77.[+229]=2,MASTERCARD;
77.[+228]=2,MASTERCARD;
77.[+227]=2,MASTERCARD;
77.[+226]=2,MASTERCARD;
77.[+225]=2,MASTERCARD;
77.[+224]=2,MASTERCARD;
77.[+223]=2,MASTERCARD;
77.[+2229]=2,MASTERCARD;
77.[+2228]=2,MASTERCARD;
77.[+2227]=2,MASTERCARD;
77.[+2226]=2,MASTERCARD;
77.[+2225]=2,MASTERCARD;
77.[+2224]=2,MASTERCARD;
77.[+2223]=2,MASTERCARD;
77.[+2222]=2,MASTERCARD;
77.[+2221]=2,MASTERCARD;
77.[+55]=2,MASTERCARD;
77.[+54]=2,MASTERCARD;
77.[+53]=2,MASTERCARD;
77.[+52]=2,MASTERCARD;
77.[+51]=2,MASTERCARD;
// JCB --------------------------------
77.[+a0000000651010]=3,JCB;
77.[+358]=3,JCB;
77.[+357]=3,JCB;
77.[+356]=3,JCB;
77.[+355]=3,JCB;
77.[+354]=3,JCB;
77.[+353]=3,JCB;
77.[+3529]=3,JCB;
77.[+3528]=3,JCB;
// DINERS -----------------------------
77.[+a0000001884443]=4,DINERS;
77.[+a0000001523010]=4,DINERS;
77.[+a0000001524010]=4,DINERS;
77.[+38]=4,DINERS;
77.[+36]=4,DINERS;
77.[+30]=4,DINERS;
// VISA -------------------------------
77.[+a0000000031010]=5,VISA;
77.[+4]=5,VISA;
// AMEX -------------------------------
77.[+a00000002501]=6,AMEX;
77.[+a0000000291010]=6,AMEX;
77.[+37]=6,AMEX;
77.[+34]=6,AMEX;
// CUP -------------------------------
77.[+a000000333010101]=12,CUP;
77.[+a000000333010102]=12,CUP;
77.[+9]=12,CUP;
77.[+8]=12,CUP;
77.[+7]=12,CUP;
77.[+62]=12,CUP;
77.[+59]=12,CUP;
77.[+58]=12,CUP;
77.[+57]=12,CUP;
77.[+56]=12,CUP;
77.[+50]=12,CUP;
77.[+3]=12,CUP;
77.[+29]=12,CUP;
77.[+28]=12,CUP;
77.[+279]=12,CUP;
77.[+278]=12,CUP;
77.[+277]=12,CUP;
77.[+276]=12,CUP;
77.[+275]=12,CUP;
77.[+274]=12,CUP;
77.[+273]=12,CUP;
77.[+2729]=12,CUP;
77.[+2728]=12,CUP;
77.[+2727]=12,CUP;
77.[+2726]=12,CUP;
77.[+2725]=12,CUP;
77.[+2724]=12,CUP;
77.[+2723]=12,CUP;
77.[+2722]=12,CUP;
77.[+2721]=12,CUP;
77.[+2220]=12,CUP;
77.[+221]=12,CUP;
77.[+220]=12,CUP;
77.[+21]=12,CUP;
77.[+20]=12,CUP;
77.[+1]=12,CUP;
// DISCOVER -------------------------------
77.[+a0000003241010]=600,DISCOVER;
77.[+65]=600,DISCOVER;
77.[+649]=600,DISCOVER;
77.[+648]=600,DISCOVER;
77.[+647]=600,DISCOVER;
77.[+646]=600,DISCOVER;
77.[+645]=600,DISCOVER;
77.[+644]=600,DISCOVER;
77.[+60119]=600,DISCOVER;
77.[+601189]=600,DISCOVER;
77.[+601188]=600,DISCOVER;
77.[+601187]=600,DISCOVER;
77.[+601186]=600,DISCOVER;
77.[+601179]=600,DISCOVER;
77.[+601178]=600,DISCOVER;
77.[+601177]=600,DISCOVER;
77.[+601174]=600,DISCOVER;
77.[+60113]=600,DISCOVER;
77.[+60112]=600,DISCOVER;
77.[+60110]=600,DISCOVER;
// **********************************************************************************************************************************************
// Terminal type 78 (CH INGENICO B&S, CRP 3.13B): 
// The card recognition is based on the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
78.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
78.[+a0000001570021]=1;
// MASTERCARD ---------------------
78.[+a0000000041010]=2;
78.[+a0000001570020]=2;
// M-Budget -----------------------
78.[a0000000000008]=2;
// JCB ---------------------------
78.[+a0000000651010]=3;
78.[+a0000001570040]=3;
// DINERS ------------------------
78.[+a0000001884443]=4;
78.[+a0000001574443]=4;
78.[+a0000001570060]=4;
78.[+a0000001523010]=4;
// VISA --------------------------
78.[+a0000000031010]=5;
78.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
78.[+a0000000032010]=10;
78.[+a0000001570031]=10;
// V PAY -------------------------------
78.[+a0000000032020]=11,V PAY;
78.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
78.[+a0000001570010]=6;
78.[+a00000002501]=6;
78.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
78.[a000000157446003]=12,CHINA UNION PAY;
78.[a000000157446004]=12,CHINA UNION PAY;
78.[a000000157446005]=12,CHINA UNION PAY;
78.[a000000157446006]=12,CHINA UNION PAY;
78.[a000000157446009]=12,CHINA UNION PAY;
78.[a0000001574460]=12,CHINA UNION PAY;
78.[a000000333010101]=12,CHINA UNION PAY;
78.[a000000333010102]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
78.[+a0000000101030]=100;
78.[+a0000001570022]=100;
78.[a0000000031030]=100;
// M-CARD ------------------------
78.[a0000001570100]=101;
78.[a0000001574450]=101;
// POSTCARD
78.[+a0000001110101]=102;
78.[+a0000001999999]=102;
78.[+a0000001999998]=102;
78.[+a0000001570051]=102;
78.[+a0000001570050]=102;
78.[+a0000001574445]=102;
78.[+a0000001574446]=102;
78.[a0000001110505]=102,POSTCARD;
// BERN CARD --------------------
78.[+a0000001570107]=103;
// ZUERI CARD -------------------
78.[+a0000001570106]=104;
// BASLER CARD ------------------
78.[+a0000001570108]=105;
// CASH -------------------------
78.[+a0000001570023]=106;
// PHARMACARD --------------------
78.[+a0000001570105]=107;
// MOEVENPICK -------------------
78.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
78.[+a0000001570103]=110;
78.[+a000000157010B]=110;
78.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
78.[a0000001574462]=137;
// MY ONE -----------------------
78.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
78.[a0000001574444]=116;
78.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
78.[a0000001574458]=134;
78.[a00000000300037562]=134;
78.[a0000000037562]=134;
// POWER CARD -------------------
78.[+a000000157010D]=117;
78.[a0000001574476]=117;
78.[a0000001574478]=117;
78.[a0000001574479]=117;
// WIR --------------------------
78.[+a000000157010C]=118;
// REKA  ------------------------------
78.[d7560000011111]=120;
78.[+a000000157010E]=120;
78.[+D7560000010101]=120;
// REKA RAIL --------------------------
78.[d7560000011112]=180;
78.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
78.[d7560000011113]=178;
78.[d7560000010102]=178;
78.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
78.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
78.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
78.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
78.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
78.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
78.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
78.[a0000001574455]=160,BONARD POINTS;
78.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK
78.[a000000157447d]=172;
78.[d7560001150001]=172;
// CCV PAY
78.[a000000157447b]=173;
// COOP EKZ
78.[37]=174;
78.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
78.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
78.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
78.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
78.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
78.[+a0000001574480]=186;
// SWISS PAY
78.[a0000008010001]=2000,SWISS PAY;
78.[37]=2000,SWISS PAY;
// SWISS PAY 2
78.[a00000015744BD]=2001,SWISS PAY 2;
78.[37]=2001,SWISS PAY 2;
//
// **********************************************************************************************************************************************
// Terminal type 79 (CH INGENICO CCV, CRP 3.13B): 
// The card recognition is based on the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
79.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
79.[+a0000001570021]=1;
// MASTERCARD ---------------------
79.[+a0000000041010]=2;
79.[+a0000001570020]=2;
// M-Budget -----------------------
79.[a0000000000008]=2;
// JCB ---------------------------
79.[+a0000000651010]=3;
79.[+a0000001570040]=3;
// DINERS ------------------------
79.[+a0000001884443]=4;
79.[+a0000001574443]=4;
79.[+a0000001570060]=4;
79.[+a0000001523010]=4;
// VISA --------------------------
79.[+a0000000031010]=5;
79.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
79.[+a0000000032010]=10;
79.[+a0000001570031]=10;
// V PAY -------------------------------
79.[+a0000000032020]=11,V PAY;
79.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
79.[+a0000001570010]=6;
79.[+a00000002501]=6;
79.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
79.[a000000157446003]=12,CHINA UNION PAY;
79.[a000000157446004]=12,CHINA UNION PAY;
79.[a000000157446005]=12,CHINA UNION PAY;
79.[a000000157446006]=12,CHINA UNION PAY;
79.[a000000157446009]=12,CHINA UNION PAY;
79.[a0000001574460]=12,CHINA UNION PAY;
79.[a000000333010101]=12,CHINA UNION PAY;
79.[a000000333010102]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
79.[+a0000000101030]=100;
79.[+a0000001570022]=100;
79.[a0000000031030]=100;
// M-CARD ------------------------
79.[a0000001570100]=101;
79.[a0000001574450]=101;
// POSTCARD
79.[+a0000001110101]=102;
79.[+a0000001999999]=102;
79.[+a0000001999998]=102;
79.[+a0000001570051]=102;
79.[+a0000001570050]=102;
79.[+a0000001574445]=102;
79.[+a0000001574446]=102;
79.[+a0000001110505]=102,POSTCARD;
79.[+a0000001110110]=102,POSTCARD;
79.[+a0000001110510]=102,POSTCARD;
79.[+a0000001110201]=102,POSTCARD;
79.[+a0000001110210]=102,POSTCARD;
// BERN CARD --------------------
79.[+a0000001570107]=103;
// ZUERI CARD -------------------
79.[+a0000001570106]=104;
// BASLER CARD ------------------
79.[+a0000001570108]=105;
// CASH -------------------------
79.[+a0000001570023]=106;
// PHARMACARD --------------------
79.[+a0000001570105]=107;
// MOEVENPICK -------------------
79.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
79.[+a0000001570103]=110;
79.[+a000000157010B]=110;
79.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
79.[a0000001574462]=137;
// MY ONE -----------------------
79.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
79.[a0000001574444]=116;
79.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
79.[a0000001574458]=134;
79.[a00000000300037562]=134;
79.[a0000000037562]=134;
// POWER CARD -------------------
79.[+a000000157010D]=117;
79.[a0000001574476]=117;
79.[a0000001574478]=117;
79.[a0000001574479]=117;
// WIR --------------------------
79.[+a000000157010C]=118;
// REKA  ------------------------------
79.[d7560000011111]=120;
79.[+a000000157010E]=120;
79.[+D7560000010101]=120;
// REKA RAIL --------------------------
79.[d7560000011112]=180;
79.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
79.[d7560000011113]=178;
79.[d7560000010102]=178;
79.[+a0000001574497]=178;
// REKA E-COMMERCE --------------------
79.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
79.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
79.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
79.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
79.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
79.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
79.[a0000001574455]=160,BONARD POINTS;
79.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK
79.[a000000157447d]=172;
79.[d7560001150001]=172;
// CCV PAY
79.[a000000157447b]=173;
// COOP EKZ
79.[37]=174;
79.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
79.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
79.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
79.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
79.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
79.[+a0000001574480]=186;
// SWISS PAY
79.[a0000008010001]=2000,SWISS PAY;
79.[37]=2000,SWISS PAY;
// SWISS PAY 2
79.[a00000015744BD]=2001,SWISS PAY 2;
79.[37]=2001,SWISS PAY 2;
// TWINT -------------------------------
79.[a000000157449e]=184,TWINT;
// Alipay ---------------------
79.[a00000015744A0]=17,AliPay;
// WECHAT PAY -----------------------
79.[a00000015744C6]=18,WeChat Pay;
//
// **********************************************************************************************************************************************
// Terminal type 80 (CH Loyalty Giftcard 1.5.4):
// The card recognition is static: GIFTCARD
// **********************************************************************************************************************************************
// GIFTCARD
80.[GIFTCARD]=183,GIFTCARD;
// GROUPCARD (Loyalty Gift Card AG)------
80.[+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 81 (Singapore Pax Interface): 
// The card recognition is based on the AID and the card name
// **********************************************************************************************************************************************
// MAESTRO ------------------------
81.[+maestro]=1;
81.[+a0000000043060]=1;
81.[+a0000001570021]=1;
// MASTERCARD ---------------------
81.[+master]=2;
81.[+mc]=2,MASTERCARD;
81.[+a0000000041010]=2;
81.[+a0000001570020]=2;
// JCB ---------------------------
81.[+jcb]=3;
81.[+a0000000651010]=3;
81.[+a0000001570040]=3;
// DINERS ------------------------
81.[+diners]=4;
81.[+a0000001884443]=4;
81.[+a0000001574443]=4;
81.[+a0000001570060]=4;
81.[+a0000001523010]=4;
// VISA --------------------------
81.[+visa]=5;
81.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
81.[+a0000000032010]=10;
81.[+a0000001570031]=10;
// V PAY (this will be routed to VISA after consenting with global pay) -------------------------------
81.[+a0000000031010]=5;
81.[+a0000000032020]=5;
81.[a0000001574452]=5;
81.[+visa debit]=5,VISA;
// AMERICAN EXPRESS --------------
81.[+amex]=6;
81.[+american express]=6;
81.[+a0000001570010]=6;
81.[+a00000002501]=6;
81.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
81.[+cup]=12;
81.[+china union pay]=12;
81.[+unionpay]=12;
81.[a000000157446003]=12,CHINA UNION PAY;
81.[a000000157446004]=12,CHINA UNION PAY;
81.[a000000157446005]=12,CHINA UNION PAY;
81.[a000000157446006]=12,CHINA UNION PAY;
81.[a000000157446009]=12,CHINA UNION PAY;
81.[a0000001574460]=12,CHINA UNION PAY;
81.[a000000333010101]=12,CHINA UNION PAY;
81.[a000000333010102]=12,CHINA UNION PAY;
81.[a000000333010102]=12,CHINA UNION PAY;
// EPS (ELECTRONIC PAYMENTS SERVICES) --------------------
81.[+eps]=1400,EPS; 
// MAESTRO CH --------------------
81.[+a0000000101030]=100;
81.[+a0000001570022]=100;
81.[a0000000031030]=100;
// M-CARD ------------------------
81.[a0000001570100]=101;
81.[a0000001574450]=101;
// POSTCARD
81.[+a0000001110101]=102;
81.[+a0000001999999]=102;
81.[+a0000001999998]=102;
81.[+a0000001570051]=102;
81.[+a0000001570050]=102;
81.[+a0000001574445]=102;
81.[+a0000001574446]=102;
//
// **********************************************************************************************************************************************
// Terminal type 82 (Malaysia Pax Interface): 
// The card recognition is based on the AID and the card name
// **********************************************************************************************************************************************
// MAESTRO ------------------------
82.[+maestro]=1;
82.[+a0000000043060]=1;
82.[+a0000001570021]=1;
// MASTERCARD ---------------------
82.[+master]=2;
82.[+mc]=2,MASTERCARD;
82.[+a0000000041010]=2;
82.[+a0000001570020]=2;
// JCB ---------------------------
82.[+jcb]=3;
82.[+a0000000651010]=3;
82.[+a0000001570040]=3;
// DINERS ------------------------
82.[+diners]=4;
82.[+a0000001884443]=4;
82.[+a0000001574443]=4;
82.[+a0000001570060]=4;
82.[+a0000001523010]=4;
// VISA --------------------------
82.[+visa]=5;
82.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
82.[+a0000000032010]=10;
82.[+a0000001570031]=10;
// V PAY (this will be routed to VISA after consenting with global pay) -------------------------------
82.[+a0000000031010]=5;
82.[+a0000000032020]=5;
82.[a0000001574452]=5;
82.[+visa debit]=5,VISA;
// AMERICAN EXPRESS --------------
82.[+amex]=6;
82.[+american express]=6;
82.[+a0000001570010]=6;
82.[+a00000002501]=6;
82.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
82.[+cup]=12;
82.[+china union pay]=12;
82.[+unionpay]=12;
82.[a000000157446003]=12,CHINA UNION PAY;
82.[a000000157446004]=12,CHINA UNION PAY;
82.[a000000157446005]=12,CHINA UNION PAY;
82.[a000000157446006]=12,CHINA UNION PAY;
82.[a000000157446009]=12,CHINA UNION PAY;
82.[a0000001574460]=12,CHINA UNION PAY;
82.[a000000333010101]=12,CHINA UNION PAY;
82.[a000000333010102]=12,CHINA UNION PAY;
82.[a000000333010102]=12,CHINA UNION PAY;
// EPS (ELECTRONIC PAYMENTS SERVICES) --------------------
82.[+eps]=1400,EPS; 
// MAESTRO CH --------------------
82.[+a0000000101030]=100;
82.[+a0000001570022]=100;
82.[a0000000031030]=100;
// M-CARD ------------------------
82.[a0000001570100]=101;
82.[a0000001574450]=101;
// POSTCARD
82.[+a0000001110101]=102;
82.[+a0000001999999]=102;
82.[+a0000001999998]=102;
82.[+a0000001570051]=102;
82.[+a0000001570050]=102;
82.[+a0000001574445]=102;
82.[+a0000001574446]=102;
//
// **********************************************************************************************************************************************
// Terminal type 83 (UK Premier Connect Protocol: 
// The card recognition is based on the payment card type
// **********************************************************************************************************************************************
// MAESTRO ------------------------
83.[+maestro]=1;
// MASTERCARD ---------------------
83.[+master]=2;
83.[+mc]=2,MASTERCARD;
// JCB ---------------------------
83.[+jcb]=3;
// DINERS ------------------------
83.[+diners]=4;
// VISA --------------------------
83.[+visa]=5;
// AMERICAN EXPRESS --------------
83.[+amex]=6;
83.[+american express]=6;
// CHINA UNION PAY --------------------
83.[+cup]=12;
83.[+china union pay]=12;
83.[+unionpay]=12;
83.[+union pay]=12;
//
// *******************************************************************************************************************************************
// Terminal type 84 (AT Hobex ZVT, derived from terminal type 55):
// *******************************************************************************************************************************************
// CHINA UNION PAY ---------------------
84.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
84.[5]=9,GIROCARD;
// EAPS -------------------------------
84.[7]=8,EAPS;
// Lastschrift ------
84.[9]=409,LASTSCHRIFT;
84.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
84.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
84.[13]=11,V PAY;
// BSW --------------------------------
84.[40]=404,BSW;
// MAESTRO ----------------------------
84.[46]=1;
// MASTERCARD -------------------------
84.[6]=2;
// DINERS -----------------------------
84.[12]=4;
// VISA -------------------------------
84.[10]=5;
// AMEX -------------------------------
84.[8]=6;
// JCB --------------------------------
84.[14]=3;
// MAESTRO DE / ELV -------------------
84.[2ELV]=410;
// MAESTRO DE / OLV -------------------
84.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
84.[2CASH]=412;
// MAESTRO DE / default ---------------
84.[2]=401;
// GELDKARTE --------------------------
84.[30]=402;
// HORNBACH Profi ---------------------
84.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
84.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
84.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
84.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
84.[42]=442,Comfort Card;
// Breuninger -------------------------
84.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
84.[+414]=414,AnniCard;
84.[+AnniCard]=414,AnniCard;
//
84.[212]=441,T-D1 XTRA;
84.[80]=441,T-D1 XTRA;
84.[+T-D1 XTRA]=441,T-D1 XTRA;
84.[+T-D1]=441,T-D1 XTRA;
84.[+XTRA]=441,T-D1 XTRA;
//
84.[191]=479,Vodafone CallYa;
84.[+Vodafone CallYa]=479,Vodafone CallYa;
84.[+Vodafone]=479,Vodafone CallYa;
84.[+CallYa]=479,Vodafone CallYa;
//
84.[190]=480,E-Plus Free&Easy;
84.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
84.[+E-Plus]=480,E-Plus Free&Easy;
84.[+Free&Easy]=480,E-Plus Free&Easy;
//
84.[188]=481,O2 LOOP;
84.[+O2 LOOP]=481,O2 LOOP;
84.[+O2]=481,O2 LOOP;
84.[+LOOP]=481,O2 LOOP;
//
84.[+431]=431,Premiere Flex;
84.[+Premiere Flex]=431,Premiere Flex;
84.[+Premiere]=431,Premiere Flex;
84.[+Flex]=431,Premiere Flex;
//
84.[189]=405,gobananas;
84.[+gobananas]=405,gobananas;
//
84.[180]=452,Paysafecard Blue;
84.[181]=452,Paysafecard Red;
84.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
84.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
84.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
84.[807]=484,SCHLECKER PREPAID;
84.[249]=486,smobil;
84.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
84.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
84.[246]=488,FONIC;
84.[206]=489,BILDmobil;
84.[205]=490,Blau.de;
84.[207]=491,Congstar;
84.[475]=492,Lebara Mobile;
84.[248]=493,Simyo;
84.[884]=494,Lycamobile;
84.[917]=495,GT mobile;
// AVS -----------------------------
84.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
84.[+6010822]=497,Alnatura Gutscheinkarte;
// 220 TLAPA-KARTE---------------------
84.[231]=220,TLAPA-KARTE;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
84.[+67]=1;
// COMFORT CARD -----------------------
84.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
84.[+5]=2;
84.[+222]=2;
84.[+223]=2;
84.[+224]=2;
84.[+225]=2;
84.[+226]=2;
84.[+227]=2;
84.[+228]=2;
84.[+229]=2;
84.[+230]=2;
84.[+231]=2;
84.[+232]=2;
84.[+233]=2;
84.[+234]=2;
84.[+235]=2;
84.[+236]=2;
84.[+237]=2;
84.[+238]=2;
84.[+239]=2;
84.[+240]=2;
84.[+241]=2;
84.[+242]=2;
84.[+243]=2;
84.[+244]=2;
84.[+245]=2;
84.[+246]=2;
84.[+247]=2;
84.[+248]=2;
84.[+249]=2;
84.[+250]=2;
84.[+251]=2;
84.[+252]=2;
84.[+253]=2;
84.[+254]=2;
84.[+284]=2;
84.[+256]=2;
84.[+257]=2;
84.[+258]=2;
84.[+259]=2;
84.[+261]=2;
84.[+262]=2;
84.[+263]=2;
84.[+264]=2;
84.[+265]=2;
84.[+266]=2;
84.[+267]=2;
84.[+268]=2;
84.[+269]=2;
84.[+270]=2;
84.[+271]=2;
84.[+272]=2;
// DINERS -----------------------------
84.[+36]=4;
// VISA -------------------------------
84.[+4]=5;
// AMEX -------------------------------
84.[+37]=6;
// JCB --------------------------------
84.[+35]=3;
//
// *******************************************************************************************************************************************
// Terminal type 85 Redsys Tpvpc 6.1 (ES):
// The card recognition is based on ISSUER-ID.
// *******************************************************************************************************************************************
// VISA -------------------------------
85.[1]=5,VISA;
// MASTERCARD--------------------------
85.[2]=2,MASTERCARD;
// DINERS -----------------------------
85.[6]=4,DINERS;
// AMEX -------------------------------
85.[8]=6,AMEX;
// JCB - Japanese Credit Banc ---------
85.[9]=3,JCB;
// CUP - China Union Pay --------------
85.[22]=12,CUP;
//
// **********************************************************************************************************************************************
// Terminal type 87 (IT INGENICO ABI Microcircuito ECR Protocol 17 Version 3.1):
// The card recognition is primarily based on the ACQUIRER ID
// **********************************************************************************************************************************************
// Setefi -----------------------------
// MAESTRO -----------------------------
87.[00000000017]=1,MAESTRO;
87.[00000088001]=1,MAESTRO;
87.[88105100066]=1,MAESTRO;
87.[00000000088]=1,MAESTRO;
// PAGOBANCOMAT (DOMESTIC MAESTRO)-----
87.[00000000000]=801,PagoBancomat EMV;
87.[00000000015]=801,PagoBancomat;
87.[15]=801,PagoBancomat;
// CARTA BANCOMAT
87.[00000010002]=801;
// AMEX -------------------------------
87.[00000000002]=6,AMEX;
87.[88105000002]=6,AMEX;
87.[00000000019]=6,AMEX EMV;
// MASTERCARD -------------------------
87.[00000000003]=2,MASTERCARD;
87.[88105100004]=2,MASTERCARD;
87.[00000000078]=2,MASTERCARD;
87.[00000000018]=2,Mastercard_S;
// VISA -------------------------------
87.[00000000004]=5,VISA;
87.[88105100003]=5,VISA;
87.[00000000008]=5,VISA;
// DINERS -----------------------------
87.[00000000005]=4,DINERS;
// JCB --------------------------------
87.[00000000006]=3,JCB;
// AGOS -------------------------------
87.[00000000008]=804,AGOS;
87.[00000000015]=809,AGOS ITA;
87.[00000000021]=810,Agos Alsolia;
// FINDOMESTIC ------------------------
87.[00000000009]=805,findomestic;
87.[00000000020]=806,findomestic Gift;
87.[00000000022]=807,findomestic Aura;
87.[00000000023]=808,findomestic NOVA;
// FIDITALIA --------------------------
87.[00000000013]=811,Fiditalia;
87.[00000000074]=812,Fiditalia PROMOZIONI;
// MONETA Visa ------------------------
87.[00000000001]=813,MONETA VISA;
// MONETA MC --------------------------
87.[00000000016]=814,MONETA MC;
// CREDIRAMA --------------------------
87.[00000000007]=815,CREDIRAMA;
// ACCORD -----------------------------
87.[00000000010]=816,Accord;
// CLESSIDRA --------------------------
87.[00000000011]=817,Clessidra;
// LINEA ------------------------------
87.[00000000012]=818,Linea;
// TOYS	-------------------------------
87.[00000000014]=819,Toys;
// MAESTRO ----------------------------
// CHINA UNION PAY --------------------
87.[00000000027]=12,CHINA UNION PAY;
// Metro ------------------------------
87.[00000000029]=820,Metro;
// UNICREDIT --------------------------
// MASTERCARD -------------------------
87.[00000000070]=2,MASTERCARD;
87.[70]=2,MASTERCARD;
// MAESTRO ----------------------------
87.[00000000066]=1,MAESTRO;
87.[66]=1,MAESTRO;
// VISA -------------------------------
87.[00000000068]=5,VISA;
87.[68]=5,VISA;
// DINERS -----------------------------
87.[00000000005]=4,DINERS;
87.[5]=4,DINERS;
// AMEX -------------------------------
87.[00000000002]=6,AMERICAN EXPRESS;
87.[00000080006]=6,AMERICAN EXPRESS;
87.[2]=6,AMERICAN EXPRESS;
// JCB --------------------------------
87.[00000000067]=3,JCB;
87.[67]=3,JCB;
// PAGOBANCOMAT (DOMESTIC MAESTRO)-----
87.[00000000060]=801,MAESTRO;
87.[60]=801,MAESTRO;
87.[00000003226]=801,MAESTRO;
87.[3226]=801,MAESTRO;
87.[00000003111]=801,MAESTRO; 
87.[3111]=801,MAESTRO;
87.[00000002008]=801,MAESTRO;
87.[2008]=801,MAESTRO;
87.[00000003500]=801,MAESTRO;
87.[3500]=801,MAESTRO;
87.[88108100001]=801,MAESTRO;
// KEY CLIENT -------------------------
87.[00000000003]=802,KEY CLIENT;
87.[3]=802,KEY CLIENT;
// CARTASI ----------------------------
87.[00000000004]=803,CARTASI;
87.[4]=803,CARTASI;
87.[00000080006]=803,CARTASI;
87.[80006]=803,CARTASI;
// AGOS -------------------------------
87.[00000000006]=804,AGOS;
87.[6]=804,AGOS;
// FINDOMESTIC ------------------------
87.[00000000009]=805,FINDOMESTIC;
87.[9]=805,FINDOMESTIC;
// BANCO DI BRESCIA
87.[00000000046]=821,BANCO DI BRESCIA;
87.[46]=821,BANCO DI BRESCIA;
//
//
// **********************************************************************************************************************************************
// Terminal type 88 (PT Adyen Interface): 
// The card recognition is based on the cardName (paymentMethod) and the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
88.[+maestro]=1;
88.[+a0000000043060]=1;
88.[+a0000001570021]=1;
88.[+a0000000050001]=1;A0000000050001
// MASTERCARD ---------------------
88.[+master]=2;
88.[+mc]=2;
88.[+a0000000041010]=2;
88.[+a0000001570020]=2;
88.[+a0000000042203]=2;
//A0000000041010    :mc_applepay
//A0000000042203    :mc_applepay
//A0000000041010    :mc_samsungpay
//A0000000042203    :mc_samsungpay
//A0000000041010    :mc_androidpay
//A0000000042203    :mc_androidpay
// JCB ---------------------------
88.[+jcb]=3;
88.[+a0000000651010]=3;
88.[+a0000001570040]=3;
// DINERS ------------------------
88.[+diners]=4;
88.[+a0000001884443]=4;
88.[+a0000001574443]=4;
88.[+a0000001570060]=4;
88.[+a0000001523010]=4;
// VISA --------------------------
88.[+visa]=5;
88.[+a0000001570030]=5;
88.[+a0000000031010]=5;
//A0000000031010    :visaalphabankbonus
//A0000000031010    :visahipotecario
//A0000000031010    :visasaraivacard
//A0000000032010    :visadankort
//A0000000980840    :visadankort
//A0000000031010    :visa_androidpay
//A0000000031010    :visa_samsungpay
//A0000000031010    :visa_applepay
// VISA ELECTRON-----------------------
88.[+a0000000032010]=10;
88.[+a0000001570031]=10;
// V PAY (this will be routed to VISA after consenting with global pay) -------------------------------
88.[+visa debit]=5;
88.[+a0000000032020]=5;
88.[a0000001574452]=5;
// AMERICAN EXPRESS --------------
88.[+amex]=6;
88.[+american express]=6;
88.[+a0000001570010]=6;
88.[+a00000002501]=6;
88.[+a0000000291010]=6;
//A00000002501      :amex
//A00000002501      :amex_samsungpay
//A00000002501      :amex_applepay
//A00000002501      :amex_androidpay
// CHINA UNION PAY --------------------
88.[+cup]=12;
88.[+union pay]=12;
88.[+a00000033301010]=12;
// DISCOVER ---------------------------
88.[+discover]=600;
88.[+a0000001523010]=600;
88.[+a0000003241010]=600;
88.[+a0000001524010]=600;
//A0000001523010    :discover_samsungpay
//A0000003241010    :discover_samsungpay
//A0000001524010    :discover_samsungpay
//A0000001523010    :discover_androidpay
//A0000003241010    :discover_androidpay
//A0000001524010    :discover_androidpay
//A0000001523010    :discover_applepay
//A0000003241010    :discover_applepay
//A0000001524010    :discover_applepay
//A0000000041010    :bijcard
//A0000000042203    :bijcard
//A0000000041010    :mcalphabankbonus
//A0000000042203    :mcalphabankbonus
//A000000176        :bcmc_mobile
//D056000666        :bcmc_mobile
//A000000176        :bcmc
//D056000666        :bcmc
// ALIPAY ---------------------------
88.[+alipay]=17,AliPay;
// WECHAT PAY -----------------------
88.[+wechatpay_pos]=18,WeChat Pay;
88.[+wechatpay]=18,WeChat Pay;
// INTERAC CARD ---------------------
88.[+interac]=1700;
88.[+A0000002771010]=1700;
// DANKORT ------------------------
88.[+a0000001211010]=1100,DANKORT
88.[+a0000001214711]=1100,DANKORT
88.[+dankort]=1100,DANKORT
//
// **********************************************************************************************************************************************
// Terminal type 89 (AT PAYLIFE + DEBIT CARD STORNO):
// The card recognition is based on the CARD NAME, given by the interface.
// (As default field CCC from p_belegzeile_1 or m_belegzeile_1, (supported by terminal software version 1.64 and greater) 
//  else m_name or, in the case of debit cards, the 3 first characters of p_text
//  else by card name)
// **********************************************************************************************************************************************
// MAESTRO -----------------------------
89.[d01]=1;
89.[d02]=1;
89.[edc]=1;
// MASTERCARD --------------------------
89.[d05]=2;
89.[c01]=2;
89.[c02]=2;
89.[KMC]=2;
89.[+mastercard]=2;
89.[+mc ]=2;
// JCB --------------------------------
89.[KJB]=3;
89.[d11]=3;
89.[c06]=3;
89.[+jcb]=3;
// DINERS -----------------------------
89.[d10]=4;
89.[c05]=4;
89.[KDI]=4;
89.[+diners]=4;
// VISA ELECTRON-----------------------
89.[+visa elec]=10;
// V PAY-------------------------------
89.[d07]=11;
89.[+v pay]=11;
89.[KVP]=11;
// VISA -------------------------------
89.[KVI]=5;
89.[d08]=5;
89.[c03]=5;
89.[+visa]=5;
89.[citibank visa]=5;
89.[hanseatic bank]=5;
89.[anzsmartpay]=5;
89.[ykb worldcard]=5;
// AMEX -------------------------------
89.[KAM]=6;
89.[d09]=6;
89.[c04]=6;
89.[+american]=6;
89.[+amex]=6;
// QUICK PIN based---------------------
89.[d06]=200;
89.[QCK]=200;
// BANKOMAT (MAESTRO AT) PIN based-----
89.[B-K]=201;
// BANKOMAT PIN based (not guaranteed)-
89.[BKN]=201;
89.[bkn]=201;
// JAPAN AIRLINES ---------------------
89.[+japan]=7;
// Euro Alliance of Payment Schemes----
89.[+eaps]=8;
// CUP (China UnionPay)----------------
89.[d12]=12;
89.[c07]=12;
89.[+cup]=12;
89.[+KCU]=12;
89.[+china]=12;
// FLOTTENKARTE (TYP 4)
89.[TYP4]=204,Flottenkarte;
// LOKALE KARTE (TYP 3)
89.[TYP3]=205;
// Inside Dornbirn
89.[V01OLP]=221,Inside Dornbirn;
89.[V01]=221,Inside Dornbirn;
89.[+Inside Dornbirn]=221,Inside Dornbirn;
//
// *********************************************************************************************************************************************
// Terminal type 90 (CH INNOCARD YOMANI / ZVT):
// *********************************************************************************************************************************************
// girocard ---------------------------
90.[5]=9,GIROCARD;
// EAPS -------------------------------
90.[7]=8,EAPS;
// Lastschrift ------
90.[9]=409,LASTSCHRIFT;
90.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
90.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
90.[13]=11,V PAY;
// BSW --------------------------------
90.[40]=404,BSW;
// POSTCARD ----------------------------
90.[69]=102,POSTFINANCE CARD;
90.[236]=102,POSTFINANCE CARD;
90.[251]=102,POSTFINANCE CARD;
// MAESTRO ----------------------------
90.[46]=1;
// MASTERCARD -------------------------
90.[6]=2;
// DINERS -----------------------------
90.[12]=4;
// VISA -------------------------------
90.[10]=5;
// AMEX -------------------------------
90.[8]=6;
// JCB --------------------------------
90.[14]=3;
// HORNBACH Profi ---------------------
90.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
90.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
90.[157]=422,HORNBACH Gutscheinkarte;
// INTERCARD GUTSCHEINKARTE------------
90.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
90.[42]=442,Comfort Card;
// Breuninger--------------------------
90.[38]=438,Breuninger;
90.[+352]=452,Paysafecard;
90.[+Paysafecard]=452,Paysafecard;
// BONCARD ----------------------------
90.[23]=160,boncard POINTS;
90.[25]=161,boncard PAY;
// SBB GESCHENKKARTE / CARTE CADEAU CFF -
90.[218]=175;
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
90.[+67]=1;
// COMFORT CARD -----------------------
90.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
90.[+5]=2;
90.[+222]=2;
90.[+223]=2;
90.[+224]=2;
90.[+225]=2;
90.[+226]=2;
90.[+227]=2;
90.[+228]=2;
90.[+229]=2;
90.[+230]=2;
90.[+231]=2;
90.[+232]=2;
90.[+233]=2;
90.[+234]=2;
90.[+235]=2;
90.[+236]=2;
90.[+237]=2;
90.[+238]=2;
90.[+239]=2;
90.[+240]=2;
90.[+241]=2;
90.[+242]=2;
90.[+243]=2;
90.[+244]=2;
90.[+245]=2;
90.[+246]=2;
90.[+247]=2;
90.[+248]=2;
90.[+249]=2;
90.[+250]=2;
90.[+251]=2;
90.[+252]=2;
90.[+253]=2;
90.[+254]=2;
90.[+255]=2;
90.[+256]=2;
90.[+257]=2;
90.[+258]=2;
90.[+259]=2;
90.[+261]=2;
90.[+262]=2;
90.[+263]=2;
90.[+264]=2;
90.[+265]=2;
90.[+266]=2;
90.[+267]=2;
90.[+268]=2;
90.[+269]=2;
90.[+270]=2;
90.[+271]=2;
90.[+272]=2;
// DINERS -----------------------------
90.[+36]=4;
// VISA -------------------------------
90.[+4]=5;
// AMEX -------------------------------
90.[+37]=6;
// JCB --------------------------------
90.[+35]=3;
// ALNATURA GUTSCHEINKARTE ------------
90.[+6010822]=497,Alnatura Gutscheinkarte;
// REKA  -------------------------------
90.[+90]=120;
// GROUPCARD (Loyalty Gift Card AG)------
90.[+a0000001574480]=186;
//
// **********************************************************************************************************************************************
// Terminal type 91 (DE VX 820 ZVT with tip capability):
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
91.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
91.[5]=9,GIROCARD;
// EAPS -------------------------------
91.[7]=8,EAPS;
// Lastschrift ------------------------
91.[9]=409,LASTSCHRIFT;
91.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
91.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
91.[13]=11,V PAY;
// BSW --------------------------------
91.[40]=404,BSW;
// MAESTRO ----------------------------
91.[46]=1;
// MASTERCARD -------------------------
91.[6]=2;
// DINERS -----------------------------
91.[12]=4;
// VISA -------------------------------
91.[10]=5;
// AMEX -------------------------------
91.[8]=6;
// JCB --------------------------------
91.[14]=3;
// MAESTRO DE / ELV -------------------
91.[2ELV]=410;
// MAESTRO DE / OLV -------------------
91.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
91.[2CASH]=412;
// MAESTRO DE / default ---------------
91.[2]=401;
// GELDKARTE --------------------------
91.[30]=402;
// IKEA FAMILY PLUS -------------------
91.[76]=403;
// HORNBACH Profi ---------------------
91.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
91.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
91.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
91.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
91.[42]=442,Comfort Card;
// Breuninger-- -----------------------
91.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
91.[+414]=414,AnniCard;
91.[+AnniCard]=414,AnniCard;
//
91.[212]=441,T-D1 XTRA;
91.[+T-D1 XTRA]=441,T-D1 XTRA;
91.[+T-D1]=441,T-D1 XTRA;
91.[+XTRA]=441,T-D1 XTRA;
//
91.[191]=479,Vodafone CallYa;
91.[+Vodafone CallYa]=479,Vodafone CallYa;
91.[+Vodafone]=479,Vodafone CallYa;
91.[+CallYa]=479,Vodafone CallYa;
//
91.[190]=480,E-Plus Free&Easy;
91.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
91.[+E-Plus]=480,E-Plus Free&Easy;
91.[+Free&Easy]=480,E-Plus Free&Easy;
//
91.[188]=481,O2 LOOP;
91.[+O2 LOOP]=481,O2 LOOP;
91.[+O2]=481,O2 LOOP;
91.[+LOOP]=481,O2 LOOP;
//
91.[+431]=431,Premiere Flex;
91.[+Premiere Flex]=431,Premiere Flex;
91.[+Premiere]=431,Premiere Flex;
91.[+Flex]=431,Premiere Flex;
//
91.[189]=405,gobananas;
91.[+gobananas]=405,gobananas;
//
91.[+352]=452,Paysafecard;
91.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
91.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
91.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
91.[807]=484,SCHLECKER PREPAID;
91.[249]=486,smobil;
91.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
91.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
91.[246]=488,FONIC;
91.[206]=489,BILDmobil;
91.[205]=490,Blau.de;
91.[207]=491,Congstar;
91.[475]=492,Lebara Mobile;
91.[248]=493,Simyo;
91.[884]=494,Lycamobile;
91.[917]=495,GT mobile;
// AVS -----------------------------
91.[253]=496,AVS Gutscheinkarte;
//
// 220 TLAPA-KARTE---------------------
91.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
91.[+67]=1;
// COMFORT CARD -----------------------
91.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
91.[+5]=2;
91.[+222]=2;
91.[+223]=2;
91.[+224]=2;
91.[+225]=2;
91.[+226]=2;
91.[+227]=2;
91.[+228]=2;
91.[+229]=2;
91.[+230]=2;
91.[+231]=2;
91.[+232]=2;
91.[+233]=2;
91.[+234]=2;
91.[+235]=2;
91.[+236]=2;
91.[+237]=2;
91.[+238]=2;
91.[+239]=2;
91.[+240]=2;
91.[+241]=2;
91.[+242]=2;
91.[+243]=2;
91.[+244]=2;
91.[+245]=2;
91.[+246]=2;
91.[+247]=2;
91.[+248]=2;
91.[+249]=2;
91.[+250]=2;
91.[+251]=2;
91.[+252]=2;
91.[+253]=2;
91.[+254]=2;
91.[+255]=2;
91.[+256]=2;
91.[+257]=2;
91.[+258]=2;
91.[+259]=2;
91.[+261]=2;
91.[+262]=2;
91.[+263]=2;
91.[+264]=2;
91.[+265]=2;
91.[+266]=2;
91.[+267]=2;
91.[+268]=2;
91.[+269]=2;
91.[+270]=2;
91.[+271]=2;
91.[+272]=2;
// DINERS -----------------------------
91.[+36]=4;
// VISA -------------------------------
91.[+4]=5;
// AMEX -------------------------------
91.[+37]=6;
// JCB --------------------------------
91.[+35]=3;
// AirPlus Company --------------------------------
91.[+AirPlus]=5000,AirPlus;
91.[+992003]=5000,AirPlus;
// Alipay ---------------------
91.[97]=17,AliPay;
// Bluecode --------------------------------
91.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 92 (HK Spectra Technologies PR608 for credit, eps and cup cards): 
// The card recognition is based on card name and AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
92.[+a0000000043060]=1,MAESTRO;
92.[+a0000001570021]=1,MAESTRO;
// MASTERCARD ---------------------
92.[master]=2,MASTERCARD;
92.[+a0000000041010]=2,MASTERCARD;
92.[+a0000001570020]=2,MASTERCARD;
// JCB ---------------------------
92.[jcb]=3,JCB;
92.[+a0000000651010]=3,JCB;
92.[+a0000001570040]=3,JCB;
// DINERS ------------------------
92.[diners]=4,DINERS;
92.[+a0000001884443]=4,DINERS;
92.[+a0000001574443]=4,DINERS;
92.[+a0000001570060]=4,DINERS;
92.[+a0000001523010]=4,DINERS;
// VISA --------------------------
92.[visa]=5,VISA;
92.[+a0000001570030]=5,VISA;
// AMERICAN EXPRESS --------------
92.[amex]=6,AMERICAN EXPRESS;
92.[+a0000001570010]=6,AMERICAN EXPRESS;
92.[+a00000002501]=6,AMERICAN EXPRESS;
92.[+a0000000291010]=6,AMERICAN EXPRESS;
// CHINA UNION PAY --------------------
92.[cup]=12,CHINA UNION PAY;
92.[a000000157446003]=12,CHINA UNION PAY;
92.[a000000157446004]=12,CHINA UNION PAY;
92.[a000000157446005]=12,CHINA UNION PAY;
92.[a000000157446006]=12,CHINA UNION PAY;
92.[a000000157446009]=12,CHINA UNION PAY;
92.[a0000001574460]=12,CHINA UNION PAY;
92.[a000000333010101]=12,CHINA UNION PAY;
92.[a000000333010102]=12,CHINA UNION PAY;
92.[a000000333010102]=12,CHINA UNION PAY;
// EPS (ELECTRONIC PAYMENTS SERVICES) --------------------
92.[eps]=1400,EPS;
//
// **********************************************************************************************************************************************
// Terminal type 93 (NETS DK): 
// The card recognition is based on card name and AID
// **********************************************************************************************************************************************
// MAESTRO DK------------------------
93.[a0000000043060]=1103,MAESTRO DK;
93.[+a0000001570021]=1,MAESTRO;
// VISA -------------------------------
93.[visa]=5,VISA;
// MASTERCARD ---------------------
93.[mastercard]=2,MASTERCARD;
93.[+a0000000041010]=2,MASTERCARD;
93.[+a0000001570020]=2,MASTERCARD;
// DANKORT ------------------------
93.[a0000001211010]=1100,DANKORT
93.[A0000001214711]=1100,DANKORT
93.[dankort]=1100,DANKORT
// JCB ---------------------------
93.[+a0000000651010]=3,JCB;
93.[+a0000001570040]=3,JCB;
93.[jcb]=3,JCB;
// VISA --------------------------
93.[+a0000000031010]=5,VISA
93.[a0000000032010]=5,VISA
93.[visa]=5,VISA;
// AMERICAN EXPRESS --------------
93.[amex]=6,AMERICAN EXPRESS;
93.[+a0000001570010]=6,AMERICAN EXPRESS;
93.[+a00000002501]=6,AMERICAN EXPRESS;
93.[+a0000000291010]=6,AMERICAN EXPRESS;
//
// *******************************************************************************************************************************************
// Terminal type 94 TransaxEFT FIS UK/IE Vx820 (O.P.I)
// The card recognition is based on the CardReportingName
// *******************************************************************************************************************************************
// MAESTRO ----------------------------
94.[+maes]=1,Maestro;
// MASTERCARD -------------------------
94.[ecmc]=2,MasterCard;
94.[+master]=2,MasterCard;
// JCB --------------------------------
94.[jcb]=3,JCB;
// DINERS -----------------------------
94.[+diner]=4,Diners Club;
// VISA -------------------------------
94.[+visa]=5,VISA;
// V PAY ------------------------------
94.[vpay]=11,V PAY;
// AMERICAN EXPRESS -------------------
94.[amex]=6,American Express;
// CHINA UNION PAY --------------------
94.[cup]=12,China Union Pay;
// BANCONTACT/MISTER CASH -------------
94.[bank]=700,Bancontact/Mister Cash;
// GIRO -------------------------------
94.[giro]=701,GIRO;
// COMORT CARD ------------------------
94.[cmfc]=704,Comfort Card;
// PRLI -------------------------------
94.[prli]=705,PRIME LINE (PRLI);
// ELECTRONISCHE POST WISSEL (EPWS) ---
94.[epws]=702,Electronische Post Wissel;
// BASIC CARD -------------------------
94.[baca]=706,Basic Card;
// HORNBACH PROFICARD NL---------------
94.[ccv]=718,Hornbach Proficard;
// Chipknip ----------------------------
94.[ck  ]=719,Chipknip;
94.[chip]=719,Chipknip;
//
// **********************************************************************************************************************************************
// **********************************************************************************************************************************************
// Terminal type 95 (PT Ingenico Remote Purchase Procotol): 
// The card recognition is based on issuer name and card number prefix
// **********************************************************************************************************************************************
// MAESTRO ------------------------
95.[+maestro]=1,MAESTRO;
// MASTERCARD ---------------------
95.[mc]=2,MASTERCARD;
95.[+master]=2,MASTERCARD;
95.[+2720]=2,MASTERCARD;
95.[+271]=2,MASTERCARD;
95.[+270]=2,MASTERCARD;
95.[+26]=2,MASTERCARD;
95.[+25]=2,MASTERCARD;
95.[+24]=2,MASTERCARD;
95.[+23]=2,MASTERCARD;
95.[+229]=2,MASTERCARD;
95.[+228]=2,MASTERCARD;
95.[+227]=2,MASTERCARD;
95.[+226]=2,MASTERCARD;
95.[+225]=2,MASTERCARD;
95.[+224]=2,MASTERCARD;
95.[+223]=2,MASTERCARD;
95.[+2229]=2,MASTERCARD;
95.[+2228]=2,MASTERCARD;
95.[+2227]=2,MASTERCARD;
95.[+2226]=2,MASTERCARD;
95.[+2225]=2,MASTERCARD;
95.[+2224]=2,MASTERCARD;
95.[+2223]=2,MASTERCARD;
95.[+2222]=2,MASTERCARD;
95.[+2221]=2,MASTERCARD;
95.[+55]=2,MASTERCARD;
95.[+54]=2,MASTERCARD;
95.[+53]=2,MASTERCARD;
95.[+52]=2,MASTERCARD;
95.[+51]=2,MASTERCARD;
// JCB ---------------------------
95.[+jcb]=3,JCB;
95.[+358]=3,JCB;
95.[+357]=3,JCB;
95.[+356]=3,JCB;
95.[+355]=3,JCB;
95.[+354]=3,JCB;
95.[+353]=3,JCB;
95.[+3529]=3,JCB;
95.[+3528]=3,JCB;
// DINERS ------------------------
95.[+diners]=4,DINERS;
95.[+38]=4,DINERS;
95.[+36]=4,DINERS;
95.[+30]=4,DINERS;
// VISA --------------------------
95.[+visa]=5,VISA;
95.[+4]=5,VISA;
// AMERICAN EXPRESS --------------
95.[+amex]=6,AMERICAN EXPRESS;
95.[+american]=6,AMERICAN EXPRESS;
95.[+37]=6,AMEX;
95.[+34]=6,AMEX;
// CHINA UNION PAY --------------------
95.[+cup]=12,CHINA UNION PAY;
95.[+unionpay]=12,CHINA UNION PAY;
95.[+chinaunionpay]=12,CHINA UNION PAY;
95.[+9]=12,CUP;
95.[+8]=12,CUP;
95.[+7]=12,CUP;
95.[+62]=12,CUP;
95.[+59]=12,CUP;
95.[+58]=12,CUP;
95.[+57]=12,CUP;
95.[+56]=12,CUP;
95.[+50]=12,CUP;
95.[+3]=12,CUP;
95.[+29]=12,CUP;
95.[+28]=12,CUP;
95.[+279]=12,CUP;
95.[+278]=12,CUP;
95.[+277]=12,CUP;
95.[+276]=12,CUP;
95.[+275]=12,CUP;
95.[+274]=12,CUP;
95.[+273]=12,CUP;
95.[+2729]=12,CUP;
95.[+2728]=12,CUP;
95.[+2727]=12,CUP;
95.[+2726]=12,CUP;
95.[+2725]=12,CUP;
95.[+2724]=12,CUP;
95.[+2723]=12,CUP;
95.[+2722]=12,CUP;
95.[+2721]=12,CUP;
95.[+2220]=12,CUP;
95.[+221]=12,CUP;
95.[+220]=12,CUP;
95.[+21]=12,CUP;
95.[+20]=12,CUP;
95.[+1]=12,CUP;
// DISCOVER -------------------------------
95.[+discover]=600,DISCOVER;
95.[+65]=600,DISCOVER;
95.[+649]=600,DISCOVER;
95.[+648]=600,DISCOVER;
95.[+647]=600,DISCOVER;
95.[+646]=600,DISCOVER;
95.[+645]=600,DISCOVER;
95.[+644]=600,DISCOVER;
95.[+60119]=600,DISCOVER;
95.[+601189]=600,DISCOVER;
95.[+601188]=600,DISCOVER;
95.[+601187]=600,DISCOVER;
95.[+601186]=600,DISCOVER;
95.[+601179]=600,DISCOVER;
95.[+601178]=600,DISCOVER;
95.[+601177]=600,DISCOVER;
95.[+601174]=600,DISCOVER;
95.[+60113]=600,DISCOVER;
95.[+60112]=600,DISCOVER;
95.[+60110]=600,DISCOVER;
// **********************************************************************************************************************************************
// Terminal type 96 (UK/IE ELAVON Smartlink iCT250): 
// The card recognition is based on card scheme name
// **********************************************************************************************************************************************
// MAESTRO ------------------------
96.[maestro]=1,MAESTRO;
96.[mc debit]=1,MAESTRO;
96.[dm]=1,MAESTRO;
96.[md]=1,MAESTRO;
// MASTERCARD ---------------------
96.[mastercard]=2,MASTERCARD;
96.[mc]=2,MASTERCARD;
// JCB ---------------------------
96.[jcb]=3,JCB;
96.[jc]=3,JCB;
// DINERS ------------------------
96.[diners]=4,DINERS;
96.[diners club]=4,DINERS;
96.[dc]=4,DINERS;
// VISA Debit --------------------------
96.[visa debit]=14,VISA Debit;
96.[vd]=14,VISA Debit;
// VISA ELECTRON --------------------------
96.[visa electron]=10,VISA ELECTRON;
96.[ve]=10,VISA ELECTRON;
// VISA --------------------------
96.[visa]=5,VISA;
96.[vi]=5,VISA;
// AMERICAN EXPRESS --------------
96.[amex]=6,AMERICAN EXPRESS;
96.[ax]=6,AMERICAN EXPRESS;
// CHINA UNION PAY --------------------
96.[cup]=12,CHINA UNION PAY;
96.[unionpay]=12,CHINA UNION PAY;
96.[cc]=12,CHINA UNION PAY;
96.[cd]=12,CHINA UNION PAY;
// GiftCard POL -----------------------
96.[giftcard pol]=21,GiftCard POL;
96.[gc]=21,GiftCard POL;
// Maestro UK -------------------------
96.[ch]=1301,Maestro UK;
// **********************************************************************************************************************************************
// Terminal type 97 (EU CCV PAD):
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
97.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
97.[5]=9,GIROCARD;
// EAPS -------------------------------
97.[7]=8,EAPS;
// Lastschrift ------------------------
97.[9]=409,LASTSCHRIFT;
97.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
97.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
97.[13]=11,V PAY;
// BSW --------------------------------
97.[40]=404,BSW;
// MAESTRO ----------------------------
97.[46]=1;
// MASTERCARD -------------------------
97.[6]=2;
// DINERS -----------------------------
97.[12]=4;
// VISA -------------------------------
97.[10]=5;
// AMEX -------------------------------
97.[8]=6;
// JCB --------------------------------
97.[14]=3;
// MAESTRO DE / ELV -------------------
97.[2ELV]=410;
// MAESTRO DE / OLV -------------------
97.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
97.[2CASH]=412;
// MAESTRO DE / default ---------------
97.[2]=401;
// GELDKARTE --------------------------
97.[30]=402;
// IKEA FAMILY PLUS -------------------
97.[76]=403;
// HORNBACH Profi ---------------------
97.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
97.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
97.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
97.[240]=440,IC-Gutscheinkarte;
// Comfort Card -----------------------
97.[42]=442,Comfort Card;
// Breuninger-- -----------------------
97.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
97.[+414]=414,AnniCard;
97.[+AnniCard]=414,AnniCard;
//
97.[212]=441,T-D1 XTRA;
97.[+T-D1 XTRA]=441,T-D1 XTRA;
97.[+T-D1]=441,T-D1 XTRA;
97.[+XTRA]=441,T-D1 XTRA;
//
97.[191]=479,Vodafone CallYa;
97.[+Vodafone CallYa]=479,Vodafone CallYa;
97.[+Vodafone]=479,Vodafone CallYa;
97.[+CallYa]=479,Vodafone CallYa;
//
97.[190]=480,E-Plus Free&Easy;
97.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
97.[+E-Plus]=480,E-Plus Free&Easy;
97.[+Free&Easy]=480,E-Plus Free&Easy;
//
97.[188]=481,O2 LOOP;
97.[+O2 LOOP]=481,O2 LOOP;
97.[+O2]=481,O2 LOOP;
97.[+LOOP]=481,O2 LOOP;
//
97.[+431]=431,Premiere Flex;
97.[+Premiere Flex]=431,Premiere Flex;
97.[+Premiere]=431,Premiere Flex;
97.[+Flex]=431,Premiere Flex;
//
97.[189]=405,gobananas;
97.[+gobananas]=405,gobananas;
//
97.[+352]=452,Paysafecard;
97.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
97.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
97.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
97.[807]=484,SCHLECKER PREPAID;
97.[249]=486,smobil;
97.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
97.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
97.[246]=488,FONIC;
97.[206]=489,BILDmobil;
97.[205]=490,Blau.de;
97.[207]=491,Congstar;
97.[475]=492,Lebara Mobile;
97.[248]=493,Simyo;
97.[884]=494,Lycamobile;
97.[917]=495,GT mobile;
// AVS -----------------------------
97.[253]=496,AVS Gutscheinkarte;
//
// 220 TLAPA-KARTE---------------------
97.[231]=220,TLAPA-KARTE; 
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
97.[+67]=1;
// COMFORT CARD -----------------------
97.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
97.[+5]=2;
97.[+222]=2;
97.[+223]=2;
97.[+224]=2;
97.[+225]=2;
97.[+226]=2;
97.[+227]=2;
97.[+228]=2;
97.[+229]=2;
97.[+230]=2;
97.[+231]=2;
97.[+232]=2;
97.[+233]=2;
97.[+234]=2;
97.[+235]=2;
97.[+236]=2;
97.[+237]=2;
97.[+238]=2;
97.[+239]=2;
97.[+240]=2;
97.[+241]=2;
97.[+242]=2;
97.[+243]=2;
97.[+244]=2;
97.[+245]=2;
97.[+246]=2;
97.[+247]=2;
97.[+248]=2;
97.[+249]=2;
97.[+250]=2;
97.[+251]=2;
97.[+253]=2;
97.[+254]=2;
97.[+255]=2;
97.[+256]=2;
97.[+257]=2;
97.[+258]=2;
97.[+259]=2;
97.[+261]=2;
97.[+262]=2;
97.[+263]=2;
97.[+264]=2;
97.[+265]=2;
97.[+266]=2;
97.[+267]=2;
97.[+268]=2;
97.[+269]=2;
97.[+270]=2;
97.[+271]=2;
97.[+272]=2;
// DINERS -----------------------------
97.[+36]=4;
// VISA -------------------------------
97.[+4]=5;
// AMEX -------------------------------
97.[+37]=6;
// JCB --------------------------------
97.[+35]=3;
// AirPlus Company --------------------------------
97.[+AirPlus]=5000,AirPlus;
97.[+992003]=5000,AirPlus;
// Alipay ---------------------
97.[97]=17,AliPay;
// Bluecode --------------------------------
97.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 98 (Verifone VX520 Global Payments VTINT protocol Malaysia):
// The card recognition is based on the AID and on the ISSUER NAME.
// **********************************************************************************************************************************************
// MASTERCARD --------------------------
98.[+a0000000041010]=2,MASTERCARD;
98.[+mastercard]=2,MASTERCARD;
98.[+mc]=2,MASTERCARD;
98.[+master]=2,MASTERCARD;
// JCB --------------------------------
98.[+a0000000651010]=3,JCB;
98.[+jcb]=3,JCB;
// DINERS -----------------------------
98.[+a0000001884443]=4,DINERS;
98.[+a0000001523010]=4,DINERS;
98.[+a0000001524010]=4,DINERS;
98.[+diners]=4,DINERS;
// VISA -------------------------------
98.[+a0000000031010]=5,VISA;
98.[+visa]=5,VISA;
// AMEX -------------------------------
98.[+a00000002501]=6,AMEX;
98.[+a0000000291010]=6,AMEX;
98.[+amex]=6,AMEX;
98.[+american]=6,AMEX;
// CUP -------------------------------
98.[+a000000333010101]=12,CUP;
98.[+a000000333010102]=12,CUP;
98.[+cup]=12,CUP;
98.[+unionpay]=12,CUP;
98.[+china]=12,CUP;
// DISCOVER -------------------------------
98.[+a0000003241010]=600,DISCOVER;
98.[+discover]=600,DISCOVER;
// RUPAY -------------------------------
98.[+rupay]=1600;
// BPI CARDS -------------------------------
98.[+bpi]=1601;
// BANCNET -------------------------------
98.[+bancnet]=1602;
// **********************************************************************************************************************************************
// Terminal type 99 (CAFIS Arch POS Linked Connection Japan):
// The card recognition is based on the AID and on the BRAND NAME.
// **********************************************************************************************************************************************
// MASTERCARD --------------------------
99.[+a0000000041010]=2,MASTERCARD;
99.[+mastercard]=2,MASTERCARD;
99.[+mc]=2,MASTERCARD;
99.[+master]=2,MASTERCARD;
// JCB --------------------------------
99.[+a0000000651010]=3,JCB;
99.[+jcb]=3,JCB;
// DINERS -----------------------------
99.[+a0000001884443]=4,DINERS;
99.[+a0000001523010]=4,DINERS;
99.[+a0000001524010]=4,DINERS;
99.[+diners]=4,DINERS;
// VISA -------------------------------
99.[+a0000000031010]=5,VISA;
99.[+visa]=5,VISA;
// AMEX -------------------------------
99.[+a00000002501]=6,AMEX;
99.[+a0000000291010]=6,AMEX;
99.[+amex]=6,AMEX;
99.[+american]=6,AMEX;
// CUP -------------------------------
99.[+a000000333010101]=12,CUP;
99.[+a000000333010102]=12,CUP;
99.[+cup]=12,CUP;
99.[+unionpay]=12,CUP;
99.[+china]=12,CUP;
// DISCOVER -------------------------------
99.[+a0000003241010]=600,DISCOVER;
99.[+discover]=600,DISCOVER;
// J-DEBIT ---------------------------
99.[j-debit]=1201,J-DEBIT;
// **********************************************************************************************************************************************
// Terminal type 100 (AliPay)
// There are currently no entries because only the one "Card Type" AliPay exists. This section is a placeholder
// **********************************************************************************************************************************************
// 100.[]=...
// **********************************************************************************************************************************************
// Terminal type 101 (WeChat)
// There are currently no entries because only the one "Card Type" WeChat exists. This section is a placeholder
// **********************************************************************************************************************************************
// 101.[]=...
// **********************************************************************************************************************************************
// Terminal type 103 (Ingenico Telium Tetra Serie CH CRP 3.13B): 
// The card recognition is based on the AID
// **********************************************************************************************************************************************
// MAESTRO ------------------------
103.[+a0000000043060]=1;   //this AID is ambiguous (could also be Maestro-CH / Pepper evaluates also the Country Code)
103.[+a0000001570021]=1;
// MASTERCARD ---------------------
103.[+a0000000041010]=2;
103.[+a0000001570020]=2;
// M-Budget -----------------------
103.[a0000000000008]=2;
// JCB ---------------------------
103.[+a0000000651010]=3;
103.[+a0000001570040]=3;
// DINERS ------------------------
103.[+a0000001884443]=4;
103.[+a0000001574443]=4;
103.[+a0000001570060]=4;
103.[+a0000001523010]=4;
// VISA --------------------------
103.[+a0000000031010]=5;
103.[+a0000001570030]=5;
// VISA ELECTRON-----------------------
103.[+a0000000032010]=10;
103.[+a0000001570031]=10;
// V PAY -------------------------------
103.[+a0000000032020]=11,V PAY;
103.[a0000001574452]=11,V PAY;
// AMERICAN EXPRESS --------------
103.[+a0000001570010]=6;
103.[+a00000002501]=6;
103.[+a0000000291010]=6;
// CHINA UNION PAY --------------------
103.[a000000157446003]=12,CHINA UNION PAY;
103.[a000000157446004]=12,CHINA UNION PAY;
103.[a000000157446005]=12,CHINA UNION PAY;
103.[a000000157446006]=12,CHINA UNION PAY;
103.[a000000157446009]=12,CHINA UNION PAY;
103.[a0000001574460]=12,CHINA UNION PAY;
103.[a000000333010101]=12,CHINA UNION PAY;
103.[a000000333010102]=12,CHINA UNION PAY;
// MAESTRO CH --------------------
103.[+a000000010103.0]=100;
103.[+a0000001570022]=100;
103.[a000000003103.0]=100;
// M-CARD ------------------------
103.[a0000001570100]=101;
103.[a0000001574450]=101;
// POSTCARD
103.[+a0000001110101]=102;
103.[+a0000001999999]=102;
103.[+a0000001999998]=102;
103.[+a0000001570051]=102;
103.[+a0000001570050]=102;
103.[+a0000001574445]=102;
103.[+a0000001574446]=102;
103.[a0000001110505]=102,POSTCARD;
103.[a0000001110110]=102,POSTCARD;
103.[a0000001110510]=102,POSTCARD;
103.[a0000001110201]=102,POSTCARD;
103.[a0000001110210]=102,POSTCARD;
// BERN CARD --------------------
103.[+a0000001570107]=103.;
// ZUERI CARD -------------------
103.[+a0000001570106]=104;
// BASLER CARD ------------------
103.[+a0000001570108]=105;
// CASH -------------------------
103.[+a0000001570023]=106;
// PHARMACARD --------------------
103.[+a0000001570105]=107;
// MOEVENPICK -------------------
103.[+a0000001570102]=108;
// JELMOLI SHOPPING BONUS CARD --
103.[+a0000001570103.]=110;
103.[+a000000157010B]=110;
103.[+a000000003000375]=110;
// JELMOLI PAYCARD --------------------
103.[a0000001574462]=137;
// MY ONE -----------------------
103.[+a0000001570104]=111;
// COOP SUPERCARD PLUS MASTERCARD-------
103.[a0000001574444]=116;
103.[a0000000049999C000020001]=116;
// COOP SUPERCARD PLUS VISA-------------
103.[a0000001574458]=134;
103.[a00000000300037562]=134;
103.[a0000000037562]=134;
// POWER CARD -------------------
103.[+a000000157010D]=117;
103.[a0000001574476]=117;
103.[a00000015744103.]=117;
103.[a0000001574479]=117;
// WIR --------------------------
103.[+a000000157010C]=118;
// REKA  ------------------------------
103.[d7560000011111]=120;
103.[+a000000157010E]=120;
103.[+D7560000010101]=120;
// REKA RAIL --------------------------
103.[d7560000011112]=180;
103.[+a0000001574495]=180;
// REKA LUNCH CARD --------------------
103.[d7560000011113]=1103.;
103.[d7560000010102]=1103.;
103.[+a0000001574497]=1103.;
// REKA E-COMMERCE --------------------
103.[a000000157448C]=181;
// JGIFT (JELMOLI)---------------------
103.[+a000000157444E]=122,J-Geschenkkarte;
// MEDIAMARKT --------------
103.[+a0000001570109]=126,Mediamarkt Card;
// M-BUDGET ---------------------------- (M-Budget is not a Mastercard when accepted by a Migros Merchant)
103.[D7560000300101]=132,M-Budget;
// SWISSCADEAU --------------------------
103.[a000000157445A]=136,SwissCadeau;
// KUNDENKARTE BONGENIE/GRIEDER ---------
103.[a000000157445E]=140,BonGenie/Grieder Card; 
// BONCARD ------------------------------
103.[a0000001574455]=160,BONARD POINTS;
103.[a000000157445B]=161,BONCARD PAY;
// LUNCH-CHECK
103.[a000000157447d]=172;
103.[d7560001150001]=172;
// CCV PAY
103.[a000000157447b]=173;
// COOP EKZ
103.[37]=174;
103.[a000000157447f]=174;
// SBB GESCHENKKARTE / CARTE CADEAU CFF
103.[a0000001574466]=175;
// CO2NEUTRAL PAY -----------------------
103.[a0000001574481]=176;
// PAYSYS PAY ---------------------------
103.[a0000001574482]=177;
// BONCARD TEST (DIREKT) --------------
103.[a0000001574486]=179;
// GROUPCARD (Loyalty Gift Card AG)------
103.[+a0000001574480]=186;
// SWISS PAY
103.[a0000008010001]=2000,SWISS PAY;
103.[37]=2000,SWISS PAY;
// SWISS PAY 2
103.[a00000015744BD]=2001,SWISS PAY 2;
103.[37]=2001,SWISS PAY 2;
// TWINT -------------------------------
103.[a000000157449e]=184,TWINT;
// Alipay ---------------------
103.[a00000015744A0]=17,AliPay;
// WECHAT PAY -----------------------
103.[a00000015744C6]=18,WeChat Pay;
//
// Terminal type 104 (Ingenico Telium Tetra Serie DE ZVT/A32.de)
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// CHINA UNION PAY ---------------------
104.[198]=12,CHINA UNION PAY;
// girocard ---------------------------
104.[5]=9,GIROCARD;
// EAPS -------------------------------
104.[7]=8,EAPS;
// Lastschrift ------
104.[9]=409,LASTSCHRIFT;
104.[9EURO]=409,LASTSCHRIFT;
// VISA electron ----------------------
104.[11]=10,VISA ELECTRON;
// V PAY ------------------------------
104.[13]=11,V PAY;
// BSW --------------------------------
104.[40]=404,BSW;
// AUSTRIA CARD -----------------------
104.[217]=217,Austria Card;
// MAESTRO ----------------------------
104.[46]=1;
// MASTERCARD -------------------------
104.[6]=2;
// DINERS -----------------------------
104.[12]=4;
// VISA -------------------------------
104.[10]=5;
// AMEX -------------------------------
104.[8]=6;
// JCB --------------------------------
104.[14]=3;
// MAESTRO DE / ELV -------------------
104.[2ELV]=410;
// MAESTRO DE / OLV -------------------
104.[2OLV]=411;
// MAESTRO DE / ec-CASH ---------------
104.[2CASH]=412;
// MAESTRO DE / default ---------------
104.[2]=401;
// GELDKARTE --------------------------
104.[30]=402;
// IKEA FAMILY PLUS -------------------
104.[76]=403;
// HORNBACH Profi ---------------------
104.[137]=420,HORNBACH Profi;
// HORNBACH Welt ----------------------
104.[138]=421,HORNBACH Welt;
// HORNBACH Welt ----------------------
104.[157]=422,HORNBACH Gutscheinkarte; 
// INTERCARD GUTSCHEINKARTE------------
104.[240]=440,IC-Gutscheinkarte;
//-------------------------------------
// Comfort Card -----------------------
104.[42]=442,Comfort Card;
// Breuninger--------------------------
104.[38]=438,Breuninger;
//
// Prepaid Phone Cards ----------------
104.[+414]=414,AnniCard;
104.[+AnniCard]=414,AnniCard;
//
104.[212]=441,T-D1 XTRA;
104.[+T-D1 XTRA]=441,T-D1 XTRA;
104.[+T-D1]=441,T-D1 XTRA;
104.[+XTRA]=441,T-D1 XTRA;
//
104.[191]=479,Vodafone CallYa;
104.[+Vodafone CallYa]=479,Vodafone CallYa;
104.[+Vodafone]=479,Vodafone CallYa;
104.[+CallYa]=479,Vodafone CallYa;
//
104.[+80]=480,E-Plus Free&Easy;
104.[+E-Plus Free&Easy]=480,E-Plus Free&Easy;
104.[+E-Plus]=480,E-Plus Free&Easy;
104.[+Free&Easy]=480,E-Plus Free&Easy;

//
104.[188]=481,O2 LOOP;
104.[+O2 LOOP]=481,O2 LOOP;
104.[+O2]=481,O2 LOOP;
104.[+LOOP]=481,O2 LOOP;
//
104.[+431]=431,Premiere Flex;
104.[+Premiere Flex]=431,Premiere Flex;
104.[+Premiere]=431,Premiere Flex;
104.[+Flex]=431,Premiere Flex;
//
104.[189]=405,gobananas;
104.[+gobananas]=405,gobananas;
//
104.[+352]=452,Paysafecard;
104.[+Paysafecard]=452,Paysafecard;
//
// TEST PINPRINTING----------------
104.[349]=482,TEST PINPRINTING;
// GESCHENKGUTSCHEIN---------------
104.[806]=483,GESCHENKGUTSCHEIN;
// SCHLECKER CARDS------------------
104.[807]=484,SCHLECKER PREPAID;
104.[249]=486,smobil;
104.[250]=487,Schlecker Zusatz;
// VITALSANA-----------------------
104.[808]=485,VITALSANA;
// MOBILE CARDS--------------------
104.[246]=488,FONIC;
104.[206]=489,BILDmobil;
104.[205]=490,Blau.de;
104.[207]=491,Congstar;
104.[475]=492,Lebara Mobile;
104.[248]=493,Simyo;
104.[884]=494,Lycamobile;
104.[917]=495,GT mobile;
// AVS -----------------------------
104.[253]=496,AVS Gutscheinkarte;
// ALNATURA GUTSCHEINKARTE ------------
104.[+6010822]=497,Alnatura Gutscheinkarte;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
104.[+67]=1;
// COMFORT CARD -----------------------
104.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
104.[+5]=2;
104.[+222]=2;
104.[+223]=2;
104.[+224]=2;
104.[+225]=2;
104.[+226]=2;
104.[+227]=2;
104.[+228]=2;
104.[+229]=2;
104.[+230]=2;
104.[+231]=2;
104.[+232]=2;
104.[+233]=2;
104.[+234]=2;
104.[+235]=2;
104.[+236]=2;
104.[+237]=2;
104.[+238]=2;
104.[+239]=2;
104.[+240]=2;
104.[+241]=2;
104.[+242]=2;
104.[+243]=2;
104.[+244]=2;
104.[+245]=2;
104.[+246]=2;
104.[+247]=2;
104.[+248]=2;
104.[+249]=2;
104.[+250]=2;
104.[+251]=2;
104.[+252]=2;
104.[+253]=2;
104.[+254]=2;
104.[+255]=2;
104.[+256]=2;
104.[+257]=2;
104.[+2104]=2;
104.[+259]=2;
104.[+261]=2;
104.[+262]=2;
104.[+263]=2;
104.[+264]=2;
104.[+265]=2;
104.[+266]=2;
104.[+267]=2;
104.[+268]=2;
104.[+269]=2;
104.[+270]=2;
104.[+271]=2;
104.[+272]=2;
// DINERS -----------------------------
104.[+36]=4;
// VISA -------------------------------
104.[+4]=5;
// AMEX -------------------------------
104.[+37]=6;
// JCB --------------------------------
104.[+35]=3;
// Alipay ---------------------
104.[97]=17,AliPay;
// Bluecode --------------------------------
104.[260]=22,BLUECODE;
// **********************************************************************************************************************************************
// Terminal type 104 (DE Ingenico ICT250 ZVT/A32.de) >>>>>> Atos Card-IDs <<<<<<
// The card recognition is based on CARD TYPE (Kartenart BMP 8A) and PAYMENT TYPE (Zahlart BMP 19). As a fall back also the card number is used.
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
//104.[21]=1;
// MASTERCARD -------------------------
//104.[1]=2;
// DINERS -----------------------------
//104.[4]=4;
// VISA -------------------------------
//104.[3]=5;
// AMEX -------------------------------
//104.[2]=6;
// MAESTRO DE / ELV -------------------
//104.[0ELV]=410;
// MAESTRO DE / OLV -------------------
//104.[0OLV]=411;
// MAESTRO DE / ec-CASH ---------------
//104.[0CASH]=412;
// MAESTRO DE / default ---------------
//104.[0]=401;
// GELDKARTE --------------------------
//104.[30]=402;
//
// For the function READ CARD the recognition must be done by the card number
// MAESTRO ----------------------------
//104.[+67]=1;
// COMFORT CARD -----------------------
//104.[+564191]=442,Comfort Card;
// MASTERCARD -------------------------
//104.[+5]=2;
// DINERS -----------------------------
//104.[+36]=4;
// VISA -------------------------------
//104.[+4]=5;
// AMEX -------------------------------
//104.[+37]=6;
// JCB --------------------------------
//104.[+35]=3;
// **********************************************************************************************************************************************
// Terminal type 105 (Landi A8, ChinaUMS):
// The card recognition is based on the Issuer Bank, International Card Organization and CreditDebitMark given by the interface.
// **********************************************************************************************************************************************
// VISA ----------------------
105.[VIS]=5,VISA;
// MASTERCARD ----------------
105.[MCC]=2,MasterCard;
// MAESTRO -------------------
105.[MAE]=1,Maestro;
// JCB -----------------------
105.[JCB]=3,JCB;
// DINERS CLUB ---------------
105.[DCC]=4,Diners Club;
// American Express ----------
105.[AMX]=6,American Express;
// Alipay --------------------
105.[+alipay]=17,AliPay;
// WeChat Pay ----------------
105.[+wechatpay]=18,WeChat Pay;
// CUP (China UnionPay)-------
105.[CUP]=12,China Union Pay;
// CUP QR Code --------------
105.[cupqrcode]=23,CUP QR CODE;
// CUP Debit --------------
105.[cupdebit]=24,CUP DEBIT;
// *******************************************************************************************************************************************
// Terminal type 113 CCV NL Vx820 (O.P.I)
// The card recognition is based on the CARDCIRCUIT
// *******************************************************************************************************************************************
// MAESTRO ----------------------------
113.[+maes]=1,Maestro;
// MASTERCARD -------------------------
113.[ecmc]=2,MasterCard;
// JCB --------------------------------
113.[jcb]=3,JCB;
// DINERS -----------------------------
113.[+diner]=4,Diners Club;
// VISA -------------------------------
113.[+visa]=5;
// V PAY ------------------------------
113.[vpay]=11,V PAY;
// AMERICAN EXPRESS -------------------
113.[amex]=6,American Express;
// CHINA UNION PAY --------------------
113.[cup]=12,China Union Pay;
// BANCONTACT/MISTER CASH -------------
113.[bank]=700,Bancontact/Mister Cash;
// GIRO -------------------------------
113.[giro]=701;
// COMORT CARD ------------------------
113.[cmfc]=704,Comfort Card;
// PRLI -------------------------------
113.[prli]=705;
// ELECTRONISCHE POST WISSEL (EPWS) ---
113.[epws]=702,Electronische Post Wissel;
// BASIC CARD -------------------------
113.[baca]=706,Basic Card;
// HORNBACH PROFICARD NL---------------
113.[ccv]=718,Hornbach Proficard;
// Chipknip ----------------------------
113.[ck  ]=719,Chipknip;
113.[chip]=719,Chipknip;
// INTERSOLVE ----------------------------
113.[isgc]=730,Intersolve Gift Card;
//
// **********************************************************************************************************************************************
// Terminal type 117 (Verifone ECR - VIRTUAL TERMINAL INTERFACE MESSAGE SPECIFICATIONS Version: BBL 10.4.12 General version):
// The card recognition is based on the AID and on the CARD NUMBER PREFIX.
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
117.[+maes]=1,Maestro;
// MASTERCARD -------------------------
117.[ecmc]=2,MasterCard;
117.[+M/C AFF]=2,MasterCard;
// JCB --------------------------------
117.[jcb]=3,JCB;
// DINERS -----------------------------
117.[+diner]=4,Diners Club;
// VISA -------------------------------
117.[+VISA]=5;
// V PAY ------------------------------
117.[vpay]=11,V PAY;
// AMERICAN EXPRESS -------------------
117.[amex]=6,American Express;
// CHINA UNION PAY --------------------
117.[cup]=12,China Union Pay;
// DISCOVER ---------------------------
117.[+discover]=600,DISCOVER;
//
// **********************************************************************************************************************************************
// Terminal type 999 (Pepper Implementation Mock):
// The card recognition is based card name string
// **********************************************************************************************************************************************
// MAESTRO ----------------------------
999.[+maestro]=1;
999.[1]=1,Maestro;
// MASTERCARD ----------------------------
999.[+mastercard]=2;
999.[2]=2,Mastercard;
// VISA ----------------------------
999.[+visa]=5;
999.[5]=5,Visa;
// VISA ELECTRON ----------------------------
999.[+elektron]=10;
999.[+electron]=10;
999.[10]=10,Visa Electron;
// AMEX ----------------------------
999.[+amex]=6;
999.[6]=6,Amex;
// DINERS ----------------------------
999.[+diners]=4;
999.[4]=4,Diners;
// JCB ----------------------------
999.[+jcb]=3;
999.[3]=3,JCB;
// V PAY ------------------------------
999.[+vpay]=11;
// CUP 
999.[12]=12,CUP;
// EuroELV
999.[413]=413,EuroELV;
// HORNBACH PROFICARD ---------------
999.[+horn]=718;
//EPS (ELECTRONIC PAYMENTS SERVICES, HK)
999.[1400]=1400,EPS;
// to simulate output cardname with amount 42XXXX00
999.[+100]=100,MAESTRO-CH;
999.[+118]=118,WIR;
999.[+184]=184,TWINT;
999.[+186]=186,GROUPCARD;
// when PayModeValue is set to Alipay or Wechat
999.[+alipay]=17;
999.[+wechat]=18;
